/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServiceSelectorBox.h
 *    A combobox widget allowing the user to select from available
 *    services providing a required interface.
 *
 * @author Christof Schröter
 * @date   2024/12/18
 */

#ifndef _MIRA_SERVICESELECTORBOX_H_
#define _MIRA_SERVICESELECTORBOX_H_

#include <QComboBox>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * This is a combobox with a method to populate it with names of services
 * providing one of a set of specified interfaces.
 * Frequently call updateServices() to keep the list of available services valid.
 */
class ServiceSelectorBox : public QComboBox
{
	Q_OBJECT

public:
	ServiceSelectorBox(QWidget* parent = nullptr);

	/// Set an interface to find services for
	void setInterface(const std::string& interface)
	{
		setInterfaces({interface});
	}

	/// Set multiple interfaces to find (union of) services for
	void setInterfaces(const std::vector<std::string>& interfaces)
	{
		mInterfaces = interfaces;
	}

	/**
	 * Set a specific service to select automatically when it
	 * is discovered for the first time
	 */
	void preferInitially(const std::string& service)
	{
		mPreferredService = service;
	}

public:
	/**
	 * Query available services for the selected interface and update the combobox.
	 * If the previously selected service is still available, it remains selected.
	 * 
	 * This should be called in the Qt thread (so it does not need locking against
	 * signal/slot selected()).
	 * 
	 * @param[in] localFirst If true, order all local services' entries before all
	 *                       remote services' entries
	 * @return True if the selection needed to change (because the previous
	 *         selection is not available anymore), false otherwise
	 */
	bool updateServices(bool localFirst = true);

	/**
	 * Get the name of the selected service.
	 *
	 * Not protected against updateServices(). If this is to be called in a
	 * different thread, the caller must handle locking.
	 */
	std::string getSelectedService();

public slots:
	/**
	 * By default, the widget automatically hides and stays hidden unless
	 * there is more than one service providing the requested interface
	 * (only visible when there is something to select - within a layout,
	 * the widget does not require any space while hidden).
	 * 
	 * This method can be used to control that behaviour - disabling autohide
	 * will make the widget visible immediately and not change again.
	 */
	void setAutoHide(bool hide);

signals:
	void selected(std::string service);

protected slots:
	void selected(int index) { emit selected(mServices[index]); }

protected:
	std::vector<std::string> mInterfaces;
	std::vector<std::string> mServices;
	std::string mPreferredService;

	bool mAutoHide = true;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
