/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file StopOnInitTest.C
 *    Test stopping/starting authority and checking status during startup.
 *
 * @author Christof Schröter
 * @date   2024/03/28
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>
#include <fw/Unit.h>

const char* argv[] = {"StopOnInitTest"};
mira::Framework fw(1,(char**)argv);

namespace mira::test {

///////////////////////////////////////////////////////////////////////////////

class StopOnInitTestUnit : public Unit
{
	MIRA_OBJECT(StopOnInitTestUnit)

public:
	StopOnInitTestUnit() : Unit(Duration::milliseconds(1000)),
	                       mInitializeStarted(false),
	                       mInitializeFinished(false),
	                       mProcessed(false),
	                       mSumCallbacks1(0),
	                       mSumCallbacks2(0)
	{
		std::cout << this << " StopOnInitTestUnit() " << std::endl;
	}

protected:
	void initialize() override
	{
		std::cout << this << "=" << getID() << std::endl;
		std::cout << getID() << " initialize()" << std::endl;

		publish<int>  (getID()+"_Published");
		subscribe<int>(getID()+"_Subscribed");
		subscribe<int>(getID()+"_SubscribedCb", &StopOnInitTestUnit::callback1, this);
		if (!isSubscribedOn(getID()+"_SubscribedCbSingleSubscribe"))
			subscribe<int>(getID()+"_SubscribedCbSingleSubscribe", &StopOnInitTestUnit::callback2, this);

		mInitializeStarted = true;
		MIRA_SLEEP(1000)

		std::cout << getID() << " initialize() finished" << std::endl;
		mInitializeFinished = true;
	}

	void process(const Timer& timer) override
	{
		std::cout << getID() << " process() " << std::endl;
		// Test: when process() runs, initialize() must have fully run before
		BOOST_CHECK(mInitializeFinished);
		mProcessed = true;
	}

	void callback1(ChannelRead<int> r)
	{
		std::cout << getID() << " callback1(" << r->value() << ")" << std::endl;
		mSumCallbacks1 += r->value();
	}

	void callback2(ChannelRead<int> r)
	{
		std::cout << getID() << " callback2(" << r->value() << ")" << std::endl;
		mSumCallbacks2 += r->value();
	}

public:
	bool mInitializeStarted;
	bool mInitializeFinished;
	bool mProcessed;
	int mSumCallbacks1;
	int mSumCallbacks2;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

MIRA_CLASS_SERIALIZATION(mira::test::StopOnInitTestUnit, mira::MicroUnit);

void checkBootupInitStatus(mira::test::StopOnInitTestUnit& unit)
{
	mira::Status::StatusMode mode = unit.getStatus();
	// the unit either:
	// - is not checked in yet (not isValid())  OR
	// - is in bootup status                    OR
	// - has finished initializing
	BOOST_CHECK(!unit.isValid() || mode == mira::Status::BOOTUP || unit.mInitializeFinished);

//	if (unit.isValid())
//		std::cout << unit.getID();
//	else
//		std::cout << &unit;
//
//	std::cout << " invalid/bootup/initialized "
//	          << !unit.isValid() << " "
//	          << (mode == mira::Status::BOOTUP) << " "
//	          << unit.mInitializeFinished << std::endl;
}

template <int N>
using Units = std::array<std::reference_wrapper<mira::test::StopOnInitTestUnit>, N>;

template<int N>
void monitorBootupInitStatus(const Units<N>& units)
{
	while (!boost::this_thread::interruption_requested()) {
		for(const auto& unit : units) {
			checkBootupInitStatus(unit);
		}
		MIRA_SLEEP(50);
	}
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(StopOnInitTest)
{
	mira::test::StopOnInitTestUnit unit1;
	mira::test::StopOnInitTestUnit unit2;
	mira::test::StopOnInitTestUnit unit3;

	// continuously monitor valid status for each unit
	boost::thread monitorThread(boost::bind(&monitorBootupInitStatus<3>,
	                                        Units<3>{{unit1, unit2, unit3}}));

	MIRA_SLEEP(1000);

	// TestUnit1: normal start
	unit1.checkin("/", "TestUnit1");
	unit1.start();

	// TestUnit2: never start -> will not initialize
	unit2.checkin("/", "TestUnit2");

	// TestUnit3: start, then stop during initialize(), then start again
	unit3.checkin("/", "TestUnit3");
	unit3.start();
	while (!unit3.mInitializeStarted) {
		MIRA_SLEEP(10)
	}
	unit3.stop(); // interrupting execution of initialize()!
	BOOST_CHECK(!unit3.mInitializeFinished);
	unit3.start();

	MIRA_SLEEP(1000);

	mira::Authority auth("/Authority");
	auto channel1 = auth.publish<int>("/TestUnit1_SubscribedCb");
	auto channel1S = auth.publish<int>("/TestUnit1_SubscribedCbSingleSubscribe");
	channel1.post(2);
	channel1S.post(2);
	MIRA_SLEEP(100);
	channel1.post(3);
	channel1S.post(3);

	auto channel3 = auth.publish<int>("/TestUnit3_SubscribedCb");
	auto channel3S = auth.publish<int>("/TestUnit3_SubscribedCbSingleSubscribe");
	channel3.post(3);
	channel3S.post(3);
	MIRA_SLEEP(100);
	channel3.post(4);
	channel3S.post(4);

	MIRA_SLEEP(3000);

	BOOST_CHECK(unit1.mProcessed);
	BOOST_CHECK(!unit2.mInitializeStarted);
	BOOST_CHECK(unit3.mProcessed);

	mira::ChannelManager& chanman = MIRA_FW.getChannelManager();

	BOOST_CHECK_EQUAL(chanman.getNrPublishers("/TestUnit1_Published"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit1_Subscribed"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit1_SubscribedCb"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit1_SubscribedCbSingleSubscribe"), 1);
	BOOST_CHECK_EQUAL(unit1.mSumCallbacks1, 5);
	BOOST_CHECK_EQUAL(unit1.mSumCallbacks2, 5);

	BOOST_CHECK_EQUAL(chanman.getNrPublishers("/TestUnit2_Published"), 0);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit2_Subscribed"), 0);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit2_SubscribedCb"), 0);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit2_SubscribedCbSingleSubscribe"), 0);

	BOOST_CHECK_EQUAL(chanman.getNrPublishers("/TestUnit3_Published"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit3_Subscribed"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit3_SubscribedCb"), 1);
	BOOST_CHECK_EQUAL(chanman.getNrSubscribers("/TestUnit3_SubscribedCbSingleSubscribe"), 1);
	BOOST_CHECK_EQUAL(unit3.mSumCallbacks1, 14);
	BOOST_CHECK_EQUAL(unit3.mSumCallbacks2, 7);

	monitorThread.interrupt();
	monitorThread.join();
}
