/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ServicePropertyTest.C
 *    Test for ServiceProperty.
 *
 * @author Christof Schröter
 */

#include <boost/test/unit_test.hpp>

#include <serialization/XMLSerializer.h>
#include <serialization/SetterNotify.h>
#include <fw/ServiceProperty.h>

using namespace std;
using namespace mira;

class ClassTest
{
public:
	ClassTest() { mService.setInterface("I"); }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Service", mService, "");
	}

	bool operator==(const ClassTest& other) const {
		return (mService == other.mService);
	}

	bool operator!=(const ClassTest& other) const {
		return !operator==(other);
	}

	ServiceProperty mService;
};

class ClassTestSetter
{
public:
	ClassTestSetter() { mService.setInterface("I"); }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Service", mService, setterNotify(mService, &ClassTestSetter::notify, this), "");
	}

	bool operator==(const ClassTestSetter& other) const {
		return (mService == other.mService);
	}

	bool operator!=(const ClassTestSetter& other) const {
		return !operator==(other);
	}

	void notify() {}

	ServiceProperty mService;
};

class ClassTestTracking
{
public:
	ClassTestTracking() { mServicePtr = NULL; }
	ClassTestTracking(bool use1) { mServicePtr = use1 ? &mService1 : &mService2; }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Service1", mService1, "");
		r.property("Service2", mService2, "");
		r.property("ServicePtr", mServicePtr, "");
	}

	bool operator==(const ClassTestTracking& other) const {
		return (mService1 == other.mService1) && (mService2 == other.mService2);
	}

	bool operator!=(const ClassTestTracking& other) const {
		return !operator==(other);
	}

	ServiceProperty mService1;
	ServiceProperty mService2;
	ServiceProperty* mServicePtr;
};

//////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( TestServiceProperty )
{
	// this test succeeds at r8631

	std::string filename = "service.xml";

	ClassTest c;
	c.mService = "S";

	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("ClassTest", c);
	xml.saveToFile(filename);

	// deserialize test class
	ClassTest c2;

	XMLDom xml2;
	xml2.loadFromFile(filename);
	XMLDeserializer s2(xml2.croot());
	s2.deserialize("ClassTest", c2);
	BOOST_CHECK(c == c2);
}

BOOST_AUTO_TEST_CASE( TestServicePropertySetter )
{
	// @ r8631, the ServiceProperty's mInterface is reset when using a setter
	// -> this test needs ServiceProperty reflection fix to succeed

	std::string filename = "service.xml";

	ClassTestSetter c;
	c.mService = "S";

	XMLDom xml;
	XMLSerializer s(xml);
	s.serialize("ClassTestSetter", c);
	xml.saveToFile(filename);

	// deserialize test class
	ClassTestSetter c2;

	XMLDom xml2;
	xml2.loadFromFile(filename);
	XMLDeserializer s2(xml2.croot());
	s2.deserialize("ClassTestSetter", c2);
	BOOST_CHECK(c == c2);
}

BOOST_AUTO_TEST_CASE( TestServicePropertyTracking )
{
	// Object tracking is disabled for ServiceProperty @ r8631, so this test fails.
	// There seems to be no requirement to disable it though (not anymore?),
	// and this test is meant to prove it works correctly when enabled.

	std::string filename = "service.xml";

	{
		ClassTestTracking c;
		c.mService1 = "S1";
		c.mService1.setInterface("I1");
		c.mService2 = "S2";
		c.mService2.setInterface("I2");

		BOOST_CHECK(c.mServicePtr == NULL);

		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize("ClassTestTracking", c);
		xml.saveToFile(filename);

		// deserialize test class
		ClassTestTracking c2;
		BOOST_CHECK(c2.mServicePtr == NULL);

		XMLDom xml2;
		xml2.loadFromFile(filename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("ClassTestTracking", c2);
		BOOST_CHECK(c == c2);
		BOOST_CHECK(c2.mServicePtr == NULL);
	}
	{
		ClassTestTracking c(true);
		c.mService1 = "S1";
		c.mService1.setInterface("I1");
		c.mService2 = "S2";
		c.mService2.setInterface("I2");

		BOOST_CHECK(c.mServicePtr == &c.mService1);

		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize("ClassTestTracking", c);
		xml.saveToFile(filename);

		// deserialize test class
		ClassTestTracking c2;
		BOOST_CHECK(c2.mServicePtr == NULL);

		XMLDom xml2;
		xml2.loadFromFile(filename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("ClassTestTracking", c2);
		BOOST_CHECK(c == c2);
		BOOST_CHECK(c2.mServicePtr == &c2.mService1);
	}
	{
		ClassTestTracking c(false);
		c.mService1 = "S1";
		c.mService1.setInterface("I1");
		c.mService2 = "S2";
		c.mService2.setInterface("I2");

		BOOST_CHECK(c.mServicePtr == &c.mService2);

		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize("ClassTestTracking", c);
		xml.saveToFile(filename);

		// deserialize test class
		ClassTestTracking c2;
		BOOST_CHECK(c2.mServicePtr == NULL);

		XMLDom xml2;
		xml2.loadFromFile(filename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("ClassTestTracking", c2);
		BOOST_CHECK(c == c2);
		BOOST_CHECK(c2.mServicePtr == &c2.mService2);
	}
}
