/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyQueryTest.C
 *    Test case for querying properties from local and remote authorities
 *
 * @author Christof Schröter
 * @date   2023/05/05
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

#include <serialization/PropertyNode.h>
#include <fw/RemoteAuthority.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(PropertyQueryTest)
{
	const char* argv[] = {"PropertyQueryTest",
	                      "${find PropertyQueryTest.xml}"};
	Framework fw(2,(char**)argv);
	fw.load();
	fw.start();

	SharedAuthority localAuthority = MIRA_FW.getAuthorityManager().getAuthority("/LocalIntPublisher");
	BOOST_REQUIRE(localAuthority != NULL);

	boost::shared_ptr<PropertyNode> localProperties = (*localAuthority).getProperties();
	BOOST_REQUIRE(localProperties != NULL);

	TypedPropertyNode<int>* localIntNode = localProperties->findChildNode("TestInt")->toTyped<int>();
	BOOST_REQUIRE(localIntNode != NULL);
	TypedPropertyNode<int>* localXNode = localProperties->findChildNode("TestXRuntime")->toTyped<int>();
	BOOST_REQUIRE(localXNode != NULL);

	BOOST_CHECK_EQUAL(localIntNode->get(), 11);

	BOOST_CHECK_THROW(localXNode->get(), XRuntime);
	try {
		int i = localXNode->get();
	}
	catch(XRuntime& ex) {
		BOOST_CHECK_EQUAL(ex.message(), "TestIntPublisherUnit property getter throws XRuntime");
	}
	BOOST_CHECK_THROW(localXNode->getAsJSON(), XRuntime);
	try {
		json::Value j = localXNode->getAsJSON();
	}
	catch(XRuntime& ex) {
		BOOST_CHECK_EQUAL(ex.message(), "TestIntPublisherUnit property getter throws XRuntime");
	}

	MIRA_SLEEP(2000);

	SharedAuthority remoteAuthority = MIRA_FW.getAuthorityManager().getAuthority("/RemoteIntPublisher");
	BOOST_REQUIRE(remoteAuthority != NULL);

	boost::shared_ptr<PropertyNode> remoteProperties = (*remoteAuthority).getProperties();
	BOOST_REQUIRE(remoteProperties != NULL);

	TypedPropertyNode<int>* remoteIntNode = remoteProperties->findChildNode("TestInt")->toTyped<int>();
	BOOST_REQUIRE(remoteIntNode != NULL);
	TypedPropertyNode<int>* remoteXNode = remoteProperties->findChildNode("TestXRuntime")->toTyped<int>();
	BOOST_REQUIRE(remoteXNode != NULL);

	// first trigger query, receive an exception, as result is not available immediately
	try {
		json::Value j = remoteIntNode->getAsJSON();
	}
	catch(XIO& ex) {
		BOOST_CHECK_EQUAL(ex.message(), "Waiting for property query");
	}
	try {
		json::Value j = remoteXNode->getAsJSON();
	}
	catch(XIO& ex) {
		BOOST_CHECK_EQUAL(ex.message(), "Waiting for property query");
	}

	// wait for query to return
	MIRA_SLEEP(200);

	// check query results
	BOOST_CHECK_EQUAL(remoteIntNode->get(), 23);

	BOOST_CHECK_THROW(remoteXNode->getAsJSON(), XRuntime);
	try {
		json::Value j = remoteXNode->getAsJSON();
	}
	catch(XRuntime& ex) {
		BOOST_CHECK_EQUAL(ex.message(), "TestIntPublisherUnit property getter throws XRuntime");
	}
}
