/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FrameworkLoaderTest.C
 *    Test cases for framework loaders.
 *
 * @author Tim Langner
 * @date   2012/04/07
 */

#include <boost/test/unit_test.hpp>

#include <transform/Pose.h>
#include <fw/Framework.h>
#include <fw/UnitManager.h>

#include <utils/PathFinder.h>

using namespace mira;

void checkNode(XMLDom::const_iterator node,
               const std::string& refContent, const std::string& refURI,
               const std::string& attributeName,
               const std::string& refAttribute)
{
	if (!refContent.empty()) {
		auto contentIt = node.content_cbegin();
		BOOST_CHECK(contentIt != node.content_cend());
		BOOST_CHECK_EQUAL(refContent, *contentIt);
	}

	if (!refURI.empty())
		BOOST_CHECK_EQUAL(refURI, node.get_attribute<std::string>("__uri__", ""));

	if (!attributeName.empty()) {
		BOOST_CHECK_EQUAL(refAttribute, node.get_attribute<std::string>(attributeName));
	}
}

void checkNode(XMLDom::const_iterator parent, const std::string& nodeName,
               const std::string& refContent, const std::string& refURI,
               const std::string& attributeName,
               const std::string& refAttribute)
{
	auto node = parent.find(nodeName);
	BOOST_REQUIRE(node != parent.cend());

	checkNode(node, refContent, refURI, attributeName, refAttribute);
}

void checkNode(XMLDom::const_iterator parent, const std::string& nodeName,
               const std::string& refContent, const std::string& refURI)
{
	checkNode(parent, nodeName, refContent, refURI, "", "");
}

void checkNode(XMLDom::const_iterator parent,
               const std::string& nodeName, size_t nth,
               const std::string& refContent, const std::string& refURI,
               const std::string& attributeName = "",
               const std::string& refAttribute = "")
{
	auto node = parent.find(nodeName, nth);
	BOOST_REQUIRE(node != parent.cend());

	checkNode(node, refContent, refURI, attributeName, refAttribute);
}

BOOST_AUTO_TEST_CASE(FrameworkLoaderTest)
{
	const char* argv[] = {"FrameworkLoaderTest",
	                      "${find FrameworkLoaderTest.xml}",
	                      "--no-colors",
	                      "--save-processed-config-annotated", "loader_test_config.xml"};
	Framework fw(5,(char**)argv);
	fw.load();
	fw.start();
	MIRA_SLEEP(1000);
	BOOST_REQUIRE(fw.getUnitManager()->getUnit("/IntPublisher") != nullptr);
	BOOST_REQUIRE(fw.getUnitManager()->getUnit("/ns1/IntSubscriber") != nullptr);
	BOOST_REQUIRE(fw.getUnitManager()->getUnit("/ns2/IntSubscriber") == nullptr);

	auto channel = fw.getChannelManager().getConcreteChannel<int>("/Int");
	ChannelRead<int> r = channel->read();
	BOOST_CHECK_GT(r->value(), 0);
	BOOST_CHECK_THROW(fw.getChannelManager().getConcreteChannel<float>("/Float"), XUnknownChannel);

	FrameworkTransformer::NodePtr rootNode = fw.getTransformer()->getNode("/RootFrame");
	FrameworkTransformer::NodePtr pose2Node = fw.getTransformer()->getNode("/Pose2Frame");
	FrameworkTransformer::NodePtr poseCov2Node = fw.getTransformer()->getNode("/PoseCov2Frame");
	FrameworkTransformer::NodePtr pose3Node = fw.getTransformer()->getNode("/Pose3Frame");
//	FrameworkTransformer::NodePtr poseCov3Node = fw.getTransformer()->getNode("/PoseCov3Frame");
	BOOST_CHECK(rootNode != NULL);
	BOOST_CHECK(pose2Node != NULL);
	BOOST_CHECK(poseCov2Node != NULL);
	BOOST_CHECK(pose3Node != NULL);
	//BOOST_CHECK(poseCov3Node != NULL);

	Pose2 p2 = fw.getTransformer()->getTransform<Pose2>(pose2Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(p2.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(p2.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(p2.phi(), deg2rad(90.0f), 0.001f);

	PoseCov2 pc2 = fw.getTransformer()->getTransform<PoseCov2>(poseCov2Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(pc2.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.phi(), deg2rad(90.0f), 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(0,0), 0.1f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(1,1), 0.2f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(2,2), 0.3f, 0.001f);
	BOOST_CHECK_CLOSE(pc2.cov(1,0), 0.0f, 0.001f);

	Pose3 p3 = fw.getTransformer()->getTransform<Pose3>(pose3Node, rootNode, Time::now());
	BOOST_CHECK_CLOSE(p3.x(), 10.0f, 0.001f);
	BOOST_CHECK_CLOSE(p3.y(), 5.0f, 0.001f);
	BOOST_CHECK_CLOSE(p3.z(), 2.5f, 0.001f);
	BOOST_CHECK_CLOSE(p3.yaw(), deg2rad(0.0f), 0.001f);
	BOOST_CHECK_CLOSE(p3.pitch(), deg2rad(90.0f), 0.001f);
	BOOST_CHECK_CLOSE(p3.roll(), deg2rad(10.0f), 0.001f);


	// analyze the saved config (URI information, in particular)
	Path configXML = findProjectPath("FrameworkLoaderTest.xml");
	Path configXMLInclude = configXML.branch_path() / "FrameworkLoaderTestInclude.xml";

	XMLDom annotatedXML;
	annotatedXML.loadFromFile("loader_test_config.xml");
	auto unitNode = annotatedXML.croot().find("unit");
	checkNode(unitNode, "StartPublish","true", configXML.string()+":46");

	checkNode(unitNode, "DummyKeep",           "X",    configXMLInclude.string()+":4");
	checkNode(unitNode, "DummyReplaceNode",    "Y",    configXML.string()+":47");
	checkNode(unitNode, "DummyAddNode",        "Y",    configXML.string()+":48");
	checkNode(unitNode, "DummyAddNode2",       "Y",    configXML.string()+":48");
	checkNode(unitNode, "DummyReplaceContent", "Y",    configXML.string()+":54");
	checkNode(unitNode, "DummyAddContent",     "Y",    configXML.string()+":55");
	checkNode(unitNode, "DummyAddContent2",    "",     configXML.string()+":56");

	auto dummy2Node = unitNode.find("DummyAddContent2");
	checkNode(dummy2Node, "DummySub",          "Y",    configXML.string()+":56");

	checkNode(unitNode, "List",                "",     configXML.string()+":49");

	auto listNode = unitNode.find("List");
	checkNode(listNode, "item", 0, "ItemBegin", configXML.string()+":60");
	checkNode(listNode, "item", 1, "ItemA", configXML.string()+":50", "dummy_attribute", "A");
	checkNode(listNode, "item", 2, "ItemBReplaced", configXML.string()+":58", "dummy_attribute", "B");
	checkNode(listNode, "item", 3, "ItemEnd", configXML.string()+":59");

	checkNode(unitNode, "List2",               "",     configXML.string()+":63");

	auto list2Node = unitNode.find("List2");
	checkNode(list2Node, "item", 0, "", configXML.string()+":63");
	auto list2Item2Node = list2Node.find("item");
	checkNode(list2Item2Node, "ItemAddNode", "Z", configXML.string()+":64");

	// stop /IntPublisher, then check that it is stopped
	auto unit = fw.getUnitManager()->getUnit("/IntPublisher");
	BOOST_CHECK(unit->isRunning());
	unit->stop();
	BOOST_CHECK(!unit->isRunning());

	// load a config with another IntPublisher
	XMLDom xml;
	xml.loadFromString("<root>"
	                   "  <namespace name=\"loadedAfterStart\">"
	                   "    <unit id=\"IntPublisher\" class=\"mira::test::TestIntPublisherUnit\" />"
	                   "  </namespace>"
	                   "</root>");
	fw.load(xml);

	// check that the new Unit is running
	auto unit2 = fw.getUnitManager()->getUnit("/loadedAfterStart/IntPublisher");
	BOOST_REQUIRE(unit2 != nullptr);
	BOOST_CHECK(unit2->isRunning());

	// check that /IntPublisher is still stopped
	BOOST_CHECK(!unit->isRunning());
}
