/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelLoader.C
 *    Implementation of a ChannelLoader.
 *
 * @author Tim Langner
 * @date   2010/10/25
 */
#include <utils/StringAlgorithms.h>

#include <fw/Framework.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Loader plugin used to modify properties of channels e.g. max slots
 */
class ChannelLoader : public ConfigurationLoaderPlugin
{
	MIRA_META_OBJECT(ChannelLoader, ("Tag", "channel"))
public:
	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* loader);
};

void ChannelLoader::parseNode(const XMLDom::const_iterator& node,
                              ConfigurationLoader* loader)
{
	auto iName = node.find_attribute("name");
	if (iName == node.attribute_end())
		MIRA_THROW(XInvalidConfig, "Error in <channel>. "
		           "No name attribute specified. "
		           "Usage <channel name=\"namespace/MyChannel\" ... />.");

	std::string  channel   = iName.value();
	ResourceName ns        = loader->getContext()["namespace"];
	std::string  channelID = MIRA_FW.getNameRegistry().resolve(channel, ns);

	try {
		if (node.find_attribute("slots") != node.attribute_end())
			MIRA_FW.getChannelManager().setMaxSlots(channelID,
			                                        node.get_attribute<std::size_t>("slots"));
		if (node.find_attribute("maxSlots") != node.attribute_end())
			MIRA_FW.getChannelManager().setMaxSlots(channelID,
			                                        node.get_attribute<std::size_t>("maxSlots"));
		if (node.find_attribute("minSlots") != node.attribute_end())
			MIRA_FW.getChannelManager().setMinSlots(channelID,
			                                        node.get_attribute<std::size_t>("minSlots"));
		if (node.find_attribute("storageDuration") != node.attribute_end())
			MIRA_FW.getChannelManager().setStorageDuration(channelID,
			         Duration::milliseconds(node.get_attribute<float>("storageDuration")*1000.0f));
		auto autoIncIt = node.find_attribute("autoIncreaseStorageDuration");
		if(autoIncIt != node.attribute_end())
			MIRA_FW.getChannelManager().setAutoIncreaseStorageDuration(channelID,
			                                                           fromString<bool>(autoIncIt.value()));
		if (node.find_attribute("type") != node.attribute_end())
			MIRA_FW.getChannelManager().setTypename(channelID,
			                                        node.get_attribute<std::string>("type"));

		MIRA_FW.getChannelManager().ensureChannel(channelID);
	} catch(Exception& ex) {
		std::cout << "Ex: " << ex.what() << std::endl;
	}


}
///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::ChannelLoader, mira::ConfigurationLoaderPlugin)
