/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DiscoverService.C
 *    Implementation of DiscoverService.h.
 *
 * @author Tim Langner
 * @date   2010/10/07
 */

#include <fw/DiscoverService.h>

#include <fw/Framework.h>
#include <fw/RemoteModule.h>
#include <fw/FrameworkDefines.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

DiscoverService::DiscoverService() :
	mSocket(static_cast<boost::asio::io_service&>(mService)),
	mEndpoint(boost::asio::ip::address::from_string("0.0.0.0"), MIRA_AUTODISCOVERY_MULTICAST_PORT),
	mMulticastAddress(boost::asio::ip::address::from_string(MIRA_AUTODISCOVERY_MULTICAST_ADDRESS))
{
	mSocket.open(mEndpoint.protocol());
	mSocket.set_option(boost::asio::ip::udp::socket::reuse_address(true));
	mSocket.bind(mEndpoint);

	// Join the multicast group.
	mSocket.set_option(boost::asio::ip::multicast::join_group(mMulticastAddress));
	startReceive();
	mService.runThreads(1, false);
}

void DiscoverService::startReceive()
{
	mSocket.async_receive_from(boost::asio::buffer((void*)&mInfo,
	                                               sizeof(AnnounceMessage)),
	                                               mSenderEndpoint,
	                                               boost::bind(&DiscoverService::handleReceive,
	                                                           this,
	                                                           boost::asio::placeholders::error,
	                                                           boost::asio::placeholders::bytes_transferred));
}

void DiscoverService::handleReceive(const boost::system::error_code& error,
                                    std::size_t bytesReceived)
{
	if ( error )
		return;

	RemoteModulePtr remote = MIRA_FW.getRemoteModule();
	if ( remote && remote->getID() != mInfo.id )
	{
		std::string ip = mSenderEndpoint.address().to_string();
		remote->onRemoteFrameworkDiscovered(ip, mInfo.port, mInfo.id);
	}
	startReceive();
}

///////////////////////////////////////////////////////////////////////////////

}
