/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelBuffer.C
 *    Implementation of ChannelBuffer.h.
 *
 * @author Erik Einhorn
 * @date   2010/09/19
 */

#include <serialization/BinaryJSONConverter.h>
#include <fw/ChannelBuffer.h>
#include <fw/Framework.h>

#include <error/Exceptions.h>

/// Uncomment this to enable integrity checking for the channel buffer (debug outputs)
//#define USE_CHANNELBUFFER_CHECK_INTEGRITY

#ifdef USE_CHANNELBUFFER_CHECK_INTEGRITY
# define CHANNELBUFFER_CHECK_INTEGRITY dbgCheckIntegrity();
#else
# define CHANNELBUFFER_CHECK_INTEGRITY
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

ChannelBufferBase::~ChannelBufferBase()
{
	MIRA_LOG(DEBUG) << "Destroying ChannelBuffer: " << this;
	CHANNELBUFFER_CHECK_INTEGRITY

	assert(isEmpty(mWriting) && "MUST CALL clear() in derived ChannelBuffer class");
	assert(isEmpty(mWaitingOrFree) && "MUST CALL clear() in derived ChannelBuffer class");
	assert(isEmpty(mBuffer) && "MUST CALL clear() in derived ChannelBuffer class");
}

void ChannelBufferBase::clear()
{
	// delete slots
	deleteSlots(mWriting);
	deleteSlots(mWaitingOrFree);
	deleteSlots(mBuffer);
}

void ChannelBufferBase::setStorageDuration(const Duration& storageDuration)
{
	boost::mutex::scoped_lock lock(mMutex);
	if(storageDuration>mStorageDuration)
		mStorageDuration=storageDuration;
}

void ChannelBufferBase::setAutoIncreaseStorageDuration(bool increase)
{
	boost::mutex::scoped_lock lock(mMutex);
	mAutoIncreaseStorageDuration = increase;
}

void ChannelBufferBase::setMinSlots(std::size_t minSlots)
{
	boost::mutex::scoped_lock lock(mMutex);
	if(minSlots>mMinSlots)
		mMinSlots = minSlots;
}

void ChannelBufferBase::setMaxSlots(std::size_t maxSlots)
{
	boost::mutex::scoped_lock lock(mMutex);

	if(maxSlots==0)
		MIRA_THROW(XInvalidParameter, "maxSlots must be larger than 0!");
	mMaxSlots=maxSlots;

	while (mSize > mMaxSlots) {
		// need unlock as requestWriteSlot will lock it
		// should be safe - no one can create extra slots,
		// and no one can can touch the slot we get
		lock.unlock();
		// must return one of the existing slots,
		// as no new ones can be allocated (we are full)
		Slot* s = requestWriteSlot();
		lock.lock();
		if (mSize <= mMaxSlots) // this could happen with setMaxSlots
			return;             // running 2 times in parallel
		{
			// remove s from slot list (mWriting)
			s->prev->next = s->next;
			s->next->prev = s->prev;
		}
		
		freeSlot(s);
		--mSize;
	}
}

std::size_t ChannelBufferBase::getSize() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return mSize;
}

std::size_t ChannelBufferBase::getMaxSlots() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return mMaxSlots;
}

std::size_t ChannelBufferBase::getMinSlots() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return mMinSlots;
}

Duration ChannelBufferBase::getStorageDuration() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return mStorageDuration;
}

bool ChannelBufferBase::isAutoIncreasingStorageDuration() const
{
	boost::mutex::scoped_lock lock(mMutex);
	return mAutoIncreaseStorageDuration;
}

///////////////////////////////////////////////////////////////////////////////

void ChannelBufferBase::resetSlot(Slot* s)
{
	s->serializedValue.clear();
	s->flags = 0;
}

ChannelBufferBase::Slot* ChannelBufferBase::requestWriteSlot()
{
	boost::mutex::scoped_lock lock(mMutex);

	CHANNELBUFFER_CHECK_INTEGRITY

	// flag that indicates if our channel is full and reached its capacity
	// if false, we can still grow
	bool bufferFull = (mSize>=mMaxSlots);

	// STEP 1
	// if we have free or waiting slots, try them first to obtain a new slot
	for(Slot* s=begin(mWaitingOrFree); s!=end(mWaitingOrFree); s=s->next)
	{
		if(s->lock.try_lock()) {
			splice(s, last(mWriting)); // insert item into writing list
			resetSlot(s);
			return s;
		}
	}

	// if we reach here we have no free slot in mWaitingOrFree, so we need to
	// grow our mBuffer or reuse old elements in mBuffer

	// STEP 2
	// check if we can reuse old elements in mBuffer

	// get the oldest and newest item
	Slot* oldest = last(mBuffer);
	Slot* newest = begin(mBuffer);

	// move backward towards newer elements
	std::size_t n=mSize;
	for(Slot* s=oldest; s!=newest; s=s->prev, --n)
	{
		// if we are full, we must start to overwrite our history that we wanted to keep

		// unless limited by maxslots, always keep one slot that is older than mStorageDuration,
		// in order to ensure coverage of that time interval
		// --> only reuse if there is a younger one still older than mStorageDuration

		if(bufferFull ||
		   ((newest->timestamp() - s->prev->timestamp() > mStorageDuration) && (n>mMinSlots)))
		{
			// timestamp of s exceeds the storage time in relation to the newest slot
			// hence, let's try to reuse it
			if(s->lock.try_lock()) {
				// IMPORTANT: in order to not disrupt the time order of our
				// elements in the ring buffer, we have to move the elements
				// oldest..s-1 at the end of the mWaitingOrFree list.
				if(s!=oldest)
					splice(s->next, oldest, last(mWaitingOrFree));

				splice(s, last(mWriting)); // insert item into writing list
				resetSlot(s);
				return s;
			}
		} else
			break; // if we reach here, we are not full and abort the loop to create a new slot in Step 3a
	}

	Slot* s = NULL;

	// STEP 3
	if(!bufferFull) {
		// we have no free slot but we can still grow some more, so do this now:
		s = allocateSlot(); // create a new slot
		++mSize;            // we created a new slot, so increase our size
		s->lock.lock();     // lock the new slot immediately

	} else {
		// if we reach here, we have no free slot and cannot grow any more, so
		// we have to BLOCK now.


		// prefer to wait for the first slot in
		// mWaitingOrFree (which should be the oldest we are waiting for)
		if(!isEmpty(mWaitingOrFree))
			s = begin(mWaitingOrFree);
		else {
			assert(!isEmpty(mBuffer));
			s = last(mBuffer);
		}

		CHANNELBUFFER_CHECK_INTEGRITY

		lock.unlock(); // free the lock before blocking
		// !!!!! HERE WE WILL BLOCK !!!!!
		// however, this is our last chance to get a slot for writing
		boost::lock(s->lock, lock); // obtain our mutex lock again before we alter the lists below
		                            // making sure to hold both or none!
		assert(lock.owns_lock());

		CHANNELBUFFER_CHECK_INTEGRITY

	}

	splice(s, last(mWriting)); // insert item into writing list
	resetSlot(s);
	return s;
}

bool ChannelBufferBase::finishWriteSlot(Slot* n, bool* dropped)
{
	boost::mutex::scoped_lock lock(mMutex);

	CHANNELBUFFER_CHECK_INTEGRITY

	// check if meta information was already created for this channel buffer
	if (!mTypeMeta && isTyped())
	{
		try
		{
			TypeMetaPtr meta = createTypeMeta(n, *MIRA_FW.getMetaDatabase());
			setTypeMeta(meta);
		}
		catch(...)
		{
			// ignore
		}
	}

	if(!n->timestamp().isValid()) {
		// discard the slot by adding it to the WaitingOrFree list
		splice(n,begin(mWaitingOrFree)->prev); // add n at the beginning of the free buffer
		n->lock.unlock();
		MIRA_THROW(XLogical, "Failed to write data with invalid timestamp");
		return false; // never reach here
	}

	Slot* newest = begin(mBuffer);

	// find the first slot that is older than n
	Slot* older = findFirstOlderSlot(n->timestamp());

	// older is the first slot, that is older than n, hence
	// the following slot is newer than n (or equal age).
	Slot* newer = older->prev;

	// make sure to handle entries with the same time stamp by ignoring the new one.
	if(newer!=end(mBuffer) && newer->timestamp() == n->timestamp()) {
		if(dropped!=NULL)
			*dropped = true;

		// discard the slot by adding it to the WaitingOrFree list
		splice(n,begin(mWaitingOrFree)->prev); // add n at the beginning of the free buffer
		n->lock.unlock();

		// NOTE: DO NOT CALL discardWriteSlot(n) here, since it will also try
		// to lock our mutex, which is already locked!!!
		// discardWriteSlot(n); // <--- NO!!
		return false;
	}

	// add n after the newer element
	splice(n, newer);
	n->lock.unlock();

	if(dropped!=NULL)
		*dropped=false;

	// the following check returns true, if the slot that was inserted is really
	// newer than all existing slots, otherwise, the slot was inserted somewhere
	// in between the existing slots and is not the newest one.
	return (older == newest); // in this case n is the new newest item (always returns true for previously empty list)
}

void ChannelBufferBase::discardWriteSlot(Slot* n)
{
	boost::mutex::scoped_lock lock(mMutex);
	splice(n,begin(mWaitingOrFree)->prev); // add n at the beginning of the free buffer
	n->lock.unlock();
}

ChannelBufferBase::Slot* ChannelBufferBase::readNewestSlot()
{
	boost::mutex::scoped_lock lock(mMutex);

	if(!isEmpty(mBuffer)) {
		Slot* newest = begin(mBuffer);
		lock.unlock();

		// here we could block, but actually shouldn't
		newest->lock.lock_shared();
		return newest;

	} else {
		// if we reach here, we have no slot in the ring buffer, but
		// there should be at least one slot that is being written to
		// however, the writing buffer could also be empty, e.g. if the
		// slots were discarded
		if(isEmpty(mWriting)) {
			return NULL;
		}

		Slot* slot = last(mWriting);
		lock.unlock();

		// here we usually will block
		slot->lock.lock_shared();

		lock.lock();

		// at this point we have locked one slot that was being written to
		// now there are 3 cases:
		//   1: we have a newer slot in the ringbuffer now, so use that
		//      one instead
		//   2: the slot we have locked was discarded but we have no other
		//      newer slot -> return NULL
		//   3: the slot is now the newest slot in the ringbuffer -> return it

		// CASE 2:
		if(isEmpty(mBuffer)) { // buffer is still empty, so slot was discarded
			slot->lock.unlock_shared();
			return NULL;
		}

		Slot* newest = begin(mBuffer);
		// CASE 1:
		if(newest!=slot) { // there is a newer slot, than our locked one
			slot->lock.unlock_shared();
			// here we could block, but actually shouldn't
			newest->lock.lock_shared();
			return newest;
		}

		// CASE 3: slot == newest
		return slot;
	}
}

ChannelBufferBase::Slot* ChannelBufferBase::readSlotAtTime(const Time& timestamp,
                                                           SlotQueryMode mode)
{
	boost::mutex::scoped_lock lock(mMutex);

	CHANNELBUFFER_CHECK_INTEGRITY

	// make sure, that we increase our storage duration automatically,
	// to fulfill future requests.
	if(!isEmpty(mBuffer)) {
		Slot* newest = begin(mBuffer);
		if (mAutoIncreaseStorageDuration)
			mStorageDuration = std::max(mStorageDuration, newest->timestamp()-timestamp);
	}

	Slot* older = findFirstOlderSlot(timestamp);
	Slot* newer = older->prev;

	Slot* s = NULL;

	switch(mode)
	{
	case OLDER_SLOT:
		s = older;
		break;
	case NEWER_SLOT:
		s = newer;
		break;
	case NEAREST_SLOT:
		if(older==end(mBuffer)) {
			s = newer;
		} else if(newer==end(mBuffer))
			s = older;
		else { // both older and newer are valid
			if( (newer->timestamp() - timestamp) <
				(timestamp - older->timestamp()))
				s=newer;
			else
				s=older;
		}
		break;
	}

	lock.unlock();

	if(s!=end(mBuffer)) {
		// !!!!! HERE WE COULD BLOCK !!!!!
		s->lock.lock_shared();
		return s;
	}
	else
		return NULL;
}

void ChannelBufferBase::readInterval(const Time& timestamp, std::size_t nrSlots,
                                     std::size_t olderSlots, std::size_t newerSlots,
                                     IntervalFillMode fillMode,
                                     std::list<Slot*>& oSlots)
{
	boost::mutex::scoped_lock lock(mMutex);

	assert(nrSlots >= olderSlots+newerSlots);

	CHANNELBUFFER_CHECK_INTEGRITY

	// make sure, that we increase our storage duration automatically,
	// to fulfill future requests.
	if(!isEmpty(mBuffer)) {
		Slot* newest = begin(mBuffer);
		if (mAutoIncreaseStorageDuration)
			mStorageDuration = std::max(mStorageDuration, newest->timestamp()-timestamp);
	}

	Slot* older = findFirstOlderSlot(timestamp);

	// |----|-----|------------|------|-----|
	// ^              ^        ^            ^
	// begin        timestamp older         end


	// move backward in time until we reach the end of the buffer,
	// or got the requested number of slots
	Slot* last = older;
	for(std::size_t i=0; i<olderSlots && last!=end(mBuffer); ++i, last=last->next) ;

	// move forward in time until we go beyond the beginning of the buffer (and reach the end),
	// or got the requested number of slots
	Slot* first = older->prev;
	for(std::size_t i=0; i<newerSlots && first!=end(mBuffer); ++i, first=first->prev) ;
	first=first->next; // we went one too far into the future

	if(first==end(mBuffer)) {
		// this should happen only if the buffer is empty!
		assert(isEmpty(mBuffer));
	}

	// now ensure the size constraint
	std::size_t left=nrSlots;
	for(Slot* s = first; s!=last; s=s->next)
		--left;

	if (fillMode == PREFER_NEWER)
	{
		while(left>0 && first!=begin(mBuffer)) {
			first=first->prev; --left;
		}
		while(left>0 && last!=end(mBuffer)) {
			last=last->next; --left;
		}
	}
	else
	{
		while(left>0 && last!=end(mBuffer)) {
			last=last->next; --left;
		}
		while(left>0 && first!=begin(mBuffer)) {
			first=first->prev; --left;
		}
	}

	if(left>0)
		MIRA_THROW(XInvalidRead, "Not enough items in ChannelBuffer to obtain "
		           "the requested interval: " << left << " out of " << nrSlots << " items missing.")


	oSlots.clear();
	// now lock all slots between first and last and put them into
	// the returned slot list
	for(Slot* s = first; s!=last; s=s->next)
	{
		s->lock.lock_shared();
		oSlots.push_back(s);
	}

	CHANNELBUFFER_CHECK_INTEGRITY
}

void ChannelBufferBase::readInterval(const Time& from, const Time& to,
                                     std::list<Slot*>& oSlots)
{
	if(from>to)
		MIRA_THROW(XInvalidParameter,
		           "'from' timestamp (" << from << ") must be the same or "
		           "in the past of 'to' (" << to << ")");

	boost::mutex::scoped_lock lock(mMutex);

	CHANNELBUFFER_CHECK_INTEGRITY

	// make sure, that we increase our storage duration automatically,
	// to fulfill future requests.
	if(!isEmpty(mBuffer)) {
		Slot* newest = begin(mBuffer);
		if (mAutoIncreaseStorageDuration)
			mStorageDuration = std::max(mStorageDuration, newest->timestamp()-from);
	}

	Slot* first = findFirstOlderSlot(to);
	// first now contains first slot that has timestamp < to or end
	// to include a possible slot with timestamp == to
	// check if next newer slot has timestamp equal to 'to' and set
	// first to this newer slot
	Slot* newer = first->prev;
	if (newer != end(mBuffer) && newer->timestamp() == to)
		first = newer;

	Slot* last  = findFirstOlderSlot(from);
	// last now contains first slot that has timestamp < from or end
	// only slots from [first|last) are included in the interval
	// so if the next newer slot has timestamp == from it would be included
	// in the interval. to exclude it check if newer slot has timestamp == from
	// and set last to this newer slot
	newer = last->prev;
	if (newer != end(mBuffer) && newer->timestamp() == from)
		last = newer;

	oSlots.clear();
	// now lock all slots between first and last and put them into
	// the returned slot list
	for(Slot* s = first; s!=last; s=s->next)
	{
		s->lock.lock_shared();
		oSlots.push_back(s);
	}

	CHANNELBUFFER_CHECK_INTEGRITY
}

///////////////////////////////////////////////////////////////////////////////

ChannelBufferBase::Slot* ChannelBufferBase::findFirstOlderSlot(const Time& timestamp,
                                                               Slot* start)
{
	CHANNELBUFFER_CHECK_INTEGRITY

	if(start==NULL)
		start = begin(mBuffer);

	// move from newest to oldest slot
	for(Slot* s=start; s!=end(mBuffer); s=s->next)
	{
		if(s->timestamp() < timestamp)
			return s;
	}
	return end(mBuffer);
}

///////////////////////////////////////////////////////////////////////////////

void ChannelBufferBase::splice(Slot* first, Slot* last, Slot* position) const
{
	assert(first!=NULL);
	assert(last!=NULL);
	assert(position!=NULL);

	CHANNELBUFFER_CHECK_INTEGRITY

	if(first->prev == position) // trying to add item at same position
		return;

	// remove from old list (will have no effect if from=to = a single item)
	first->prev->next = last->next;
	last->next->prev = first->prev;

	// insert into new list
	first->prev = position;
	last->next  = position->next;
	last->next->prev = last;
	position->next = first;

	CHANNELBUFFER_CHECK_INTEGRITY
}


void ChannelBufferBase::deleteSlots(ListItem& list)
{
	CHANNELBUFFER_CHECK_INTEGRITY
	for(Slot* s=begin(list); s!=end(list); /* we are iterating in the for-body */ )
	{
		Slot* d = s;
		s = s->next; // go to next slot
		freeSlot(d); // delete the slot
	}
	// set list empty
	list.next = (Slot*)(&list);
	list.prev = (Slot*)(&list);
}

///////////////////////////////////////////////////////////////////////////////

std::string ChannelBufferBase::dbgDumpList(ListItem& list, bool brief)
{
	std::stringstream dbgOut;

	if(!brief)
		dbgOut << std::endl;

	for(Slot* s=begin(list); s!=end(list); s=s->next)
	{
		if(s->lock.try_lock_shared()) {
			dbgOut << "r";
			s->lock.unlock_shared();
		} else
			dbgOut << "!"; // cannot read

		if(s->lock.try_lock()) {
			dbgOut << "w";
			s->lock.unlock();
		} else
			dbgOut << "!"; // cannot write

		if(brief)
			dbgOut << ",";
		else
			dbgOut << " @ " << s << " time: "<< s->timestamp() << std::endl;
	}

	return dbgOut.str();
}

void ChannelBufferBase::dbgDump(const std::string& prefix, bool brief)
{
	boost::mutex::scoped_lock lock(mMutex);
	MIRA_LOG(NOTICE) << "[" << prefix << "]: " << mSize << std::endl
			<< "buffer:  " << dbgDumpList(mBuffer, brief) << std::endl
			<< "waiting: " << dbgDumpList(mWaitingOrFree, brief) << std::endl
			<< "writing: " << dbgDumpList(mWriting, brief) << std::endl;
}

void ChannelBufferBase::dbgCheckListIntegrity(const ListItem& list)
{
	const Slot* s0 = (const Slot*)&list;
	const Slot* s = s0;

	do {
		assert(s->prev->next==s);
		assert(s->next->prev==s);
		s=s->next;
	} while(s!=s0);
}

void ChannelBufferBase::dbgCheckIntegrity()
{
#ifndef NDEBUG
	dbgCheckListIntegrity(mBuffer);
	dbgCheckListIntegrity(mWaitingOrFree);
	dbgCheckListIntegrity(mWriting);
#endif
}

///////////////////////////////////////////////////////////////////////////////

void ChannelBuffer<void>::setTypename(const Typename& name)
{
	mTypename = name;

	// try to obtain the type meta from the database
	auto db = MIRA_FW.getMetaDatabase();
	auto it = db->find(name);
	TypeMetaPtr meta;
	if(it!=db->end())
	{
		meta.reset(new TypeMeta);
		meta->setClassType(name);
		setTypeMeta(meta);
	}
	else
	{
		// check if type is an atomic type
		if( name=="char" ||
			name=="signed char" ||
			name=="unsigned char" ||
			name=="short" ||
			name=="unsigned short" ||
			name=="int" ||
			name=="unsigned int" ||
			name=="unsigned int64" ||
			name=="float" ||
			name=="double" ||
			name=="std::string" ||
			name=="bool" ||
			name=="enum" ||
			name=="int8" ||
			name=="uint8" ||
			name=="int16" ||
			name=="uint16" ||
			name=="int32" ||
			name=="uint32" ||
			name=="int64" ||
			name=="uint64" ||
			name=="string")
		{
			meta.reset(new TypeMeta);
			meta->setAtomicType(name);
			setTypeMeta(meta);
		}
	}
}

void ChannelBuffer<void>::readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue)
{
	if (!getTypeMeta())
		MIRA_THROW(XNotImplemented, "readJSON is not supported for untyped channels that have no meta information");
	BinaryJSONConverter::binaryToJSON(s->serializedValue, false,
	                                  *getTypeMeta().get(), *MIRA_FW.getMetaDatabase(),
	                                  oValue);
}

void ChannelBuffer<void>::readJSON(ChannelBufferBase::Slot* s, JSONValue& oValue, JSONSerializer& serializer)
{
	MIRA_THROW(XNotImplemented,
	           "readJSON on untyped channel must use binary-to-JSON conversion, cannot use specified serializer");
}

void ChannelBuffer<void>::writeJSON(ChannelBufferBase::Slot* s, const JSONValue& value)
{
	if (!getTypeMeta())
		MIRA_THROW(XNotImplemented, "writeJSON is not supported for untyped channels that have no meta information");
	BinaryJSONConverter::JSONToBinary(value,
	                                  *getTypeMeta().get(), *MIRA_FW.getMetaDatabase(),
	                                  s->serializedValue, false);
}

void ChannelBuffer<void>::writeJSON(ChannelBufferBase::Slot* s, JSONDeserializer& deserializer)
{
	MIRA_THROW(XNotImplemented,
	           "writeJSON on untyped channel must use JSON-to-binary conversion, cannot use specified deserializer");
}

void ChannelBuffer<void>::writeXML(ChannelBufferBase::Slot* s, const XMLDom::const_iterator& node)
{
	MIRA_THROW(XNotImplemented, "writeXML is not supported for untyped channels");
}

///////////////////////////////////////////////////////////////////////////////

int PolymorphicChannelBuffer::getTypeId() const {
	if(mFixatedClass==NULL)
		MIRA_THROW(XLogical, "Cannot obtain typeid of an empty polymorphic"
		           " channel where the derived type is not yet known");
	return mFixatedClass->getTypeId();

}

Typename PolymorphicChannelBuffer::getTypename() const {
	if(mFixatedClass==NULL)
		MIRA_THROW(XLogical, "Cannot obtain typename of an empty polymorphic"
		           " channel where the derived type is not yet known");
	return mFixatedClass->getTypename() + "*";
}

ChannelBufferBase::Slot* PolymorphicChannelBuffer::allocateSlot() {
	assert(mFixatedClass!=NULL && "The class of a polymorphic channel should "
	       "have been set using fixateType() before allocating a slot");
	Slot* slot = new Slot;
	assert(slot->data.value() == NULL);
	// allocate the polymorphic object and set it to the slots value pointer
	slot->data = mFixatedClass->newInstance();
	slot->timestampPtr = &slot->data.timestamp; // "connect" the timestamp ptr
	return slot;
}

void PolymorphicChannelBuffer::freeSlot(ChannelBufferBase::Slot* s) {
	// casting s to our slot type ensures that the correct destructor is called
	Slot* slot = castSlot(s);
	delete slot;
}

void PolymorphicChannelBuffer::fixateType()
{
	// make sure our derived type has not changed since last call
	assert((mFixatedClass==NULL || mFixatedClass==mMostDerivedClass) &&
	       "Must not change type when calling fixateType() multiple times");

	assert(mMostDerivedClass!=NULL &&
	       "Should never happen since we always should have a type set "
	       "as soon as we are created");

	// fixate the type of this buffer to T by storing the pointer to the class of T
	if(mFixatedClass==NULL) {
		mFixatedClass = mMostDerivedClass;
	}
}

void PolymorphicChannelBuffer::writeSerializedValue(ChannelBufferBase::Slot* s, Buffer<uint8> data)
{
	// call the base class to do the writing of the serialized data
	Base::writeSerializedValue(s, std::move(data));

	// but afterwards perform some checks

	// get the object that we have deserialized above
	// we can safely cast here, since s is an object of our Slot type
	Slot* slot = static_cast<Slot*>(s);
	Object* obj = slot->data.value();

	if(obj==NULL) // if NULL was deserialized, we cannot check anything
		return;

	const Class* objClass = &obj->getClass();

	if(mFixatedClass==NULL) {
		// if we have no fixated class yet, then the type of the deserialized
		// data will determine the fixed type of this channel
		try {
			// promote to that type first (this could result in an exception, see below)
			promote(objClass);
		} catch(Exception& ex) {
			// if we cannot promote, add some explaining error message
			MIRA_RETHROW(ex, "Cannot write serialized object into "
			             "polymorphic ChannelBuffer, since the class of "
			             "the serialized object does not match the type of "
			             "the ChannelBuffer");
		}
		fixateType(); // fixate us to the type we have written
	} else {
		// otherwise, we have to check if the class types match
		if(mFixatedClass!=objClass)
			MIRA_THROW(XBadCast, "Cannot write serialized object into "
			           "polymorphic ChannelBuffer, since the class of the "
			           "serialized object does not match the type of the "
			           "ChannelBuffer");
	}
}
//@}

/// Promotes the polymorphic type of the channel to the given class type
void PolymorphicChannelBuffer::promote(const Class* promotionClass)
{
	// if we already have some type, so do some necessary type checks first
	// and bail out if there's nothing to do
	if(mMostDerivedClass!=NULL) {

		// check if current derived class and the class to promote to are the same
		if(mMostDerivedClass==promotionClass)
			return; // if so, we do not need to do anything, since the types are the same

		// if we are already more derived than the class to promote to, there's nothing to promote to
		if(mMostDerivedClass->isDerivedFrom(promotionClass))
			return;

		// if we reach here, promotedClass should be derived from our current type, check this now
		if(!promotionClass->isDerivedFrom(mMostDerivedClass))
			MIRA_THROW(XBadCast, "Invalid promotion from typed polymorphic "
			           "ChannelBuffer. The specified type '"
			           << promotionClass->getIdentifier()
			           << "'is not derived from the type '"
			           << mMostDerivedClass->getIdentifier()
			           << "' of the existing channel buffer.");

		if (mFixatedClass)
			MIRA_THROW(XBadCast, "Invalid promotion of typed polymorphic ChannelBuffer to '"
			           << promotionClass->getIdentifier()
			           << "'. The type has already been fixated to  '"
			           << mMostDerivedClass->getIdentifier()
			           << "'.");


	} else {
		// we have no class type yet

		// we should also have no fixated class (in fixateType() where
		// mFixatedClass is set also mMostDerivedClass is set...)
		assert(mFixatedClass==NULL && "We should not have a fixated class "
		       "if we have no derived class type, otherwise something went wrong "
		       "in fixateType()");
	}

	// if we reach here we can safely promote us and we are done
	mMostDerivedClass = promotionClass;
}

template class TypedChannelBufferBase<Object*>;

///////////////////////////////////////////////////////////////////////////////

}
