/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Authority.hpp
 *    Implementation header for all template functions of the authority classes.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/10/08
 */

#ifndef _MIRA_AUTHORITYIMPL_H_
#define _MIRA_AUTHORITYIMPL_H_

#include <fw/Framework.h>
#include <fw/ServiceCall.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template <typename F>
inline void Authority::addImmediateHandlerFunction(F&& fn)
{
	validate();
	mRuntime->getMainDispatcher().addImmediateHandlerFunction(static_cast<F&&>(fn), this);
}

template <typename F>
inline void Authority::addFinalizeHandlerFunction(F&& fn)
{
	validate();
	mRuntime->getMainDispatcher().addFinalizeHandlerFunction(static_cast<F&&>(fn), this);
}


template<typename Class>
inline TimerPtr Authority::createTimer(Duration period,
                                       void (Class::*f)(const Timer&), Class* obj,
                                       bool oneshot)
{
	return createTimer(period, boost::bind(f, obj, _1), oneshot);
}

template<typename Class>
inline TimerPtr Authority::createTimer(Duration period,
                                       void (Class::*f)(const Timer&),
                                       bool oneshot)
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::createTimer(..., Class::member, ...)' on an object that is "
	       "not of type Class (you may need to provide a pointer to the correct object)!");
	return createTimer(period, f, obj, oneshot);
}

template<typename Class>
inline TimerPtr Authority::createTimer(Duration period, Duration tolerance,
                                       void (Class::*f)(const Timer&), Class* obj,
                                       bool oneshot)
{
	return createTimer(period, tolerance, boost::bind(f, obj, _1), oneshot);
}

template<typename Class>
inline TimerPtr Authority::createTimer(Duration period, Duration tolerance,
                                       void (Class::*f)(const Timer&),
                                       bool oneshot)
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::createTimer(..., Class::member, ...)' on an object that is "
	       "not of type Class (you may need to provide a pointer to the correct object)!");
	return createTimer(period, tolerance, f, obj, oneshot);
}

template<typename Class>
inline TimerPtr Authority::createTimer(Time time,
                                       void (Class::*f)(const Timer&), Class* obj)
{
	return createTimer(time, boost::bind(f, obj, _1));
}

template<typename Class>
inline TimerPtr Authority::createTimer(Time time, void (Class::*f)(const Timer&))
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::createTimer(..., Class::member, ...)' on an object that is "
	       "not of type Class (you may need to provide a pointer to the correct object)!");
	return createTimer(time, f, obj);
}

template<typename T>
inline Channel<T> Authority::publish(const std::string& iChannelID,
                                     const Typename& type)
{
	validate();

	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	// insert name into global/local name map
	insertChannelNameMapping(mPublishedChannelNames, iChannelID, channelID);

	// check if already published
	if ( MIRA_FW.getChannelManager().hasPublished(getGlobalID(), channelID) )
	{
		Channel<T> channel = getChannel<T>(iChannelID);
		// check if types of the channels match
		if ( channel.getTypename() != type )
			MIRA_THROW(XLogical, "Channel typenames do not match '" << type
			           << "' != '" << channel.getTypename() << "'");
		// already published with matching types so return that channel
		return channel;
	}

	// If we are a remote authority we will store the information about us
	// publishing that channel with the internal flag set to true. Other instances
	// asking the channel manager about non internal published channels will not
	// see us as a publisher to avoid cyclic dependencies.
	ConcreteChannel<T>* channel = MIRA_FW.getChannelManager().publish<T>(channelID,
	                                                                     getGlobalID(),
	                                                                     mIsInvisiblePublisherSubscriber, type);

	// set access rights
	setChannelWriteAccess(channelID);
	return toProxy(channel);
}

template<typename T>
inline Channel<T> Authority::subscribe(const std::string& iChannelID,
                                       const Duration& storageDuration)
{
	validate();

	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	// insert name into global/local name map
	insertChannelNameMapping(mSubscribedChannelNames, iChannelID, channelID);

	// check if already subscribed
	if ( MIRA_FW.getChannelManager().isSubscribedOn(getGlobalID(), channelID) )
		return getChannel<T>(iChannelID);

	// set access rights
	setChannelReadAccess(channelID);

	// If we are a remote authority we will store the information about us
	// subscribing to that channel with the internal flag set to true. Other
	// instances asking the channel manager about non internal subscribers for
	// channels will not see us as a subscriber to avoid cyclic dependencies.
	ConcreteChannel<T>* channel = MIRA_FW.getChannelManager().subscribe<T>(channelID,
	                                                                       getGlobalID(),
	                                                                       storageDuration,
	                                                                       mIsInvisiblePublisherSubscriber);

	// add us as subscriber
	channel->addSubscriber();
	return toProxy(channel);
}

template<typename T>
inline Channel<T> Authority::subscribe(const std::string& iChannelID, 
                                       boost::function<void (ChannelRead<T>)> fn, 
                                       const Duration& storageDuration, 
                                       bool independentThread)
{
	validate();

	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	// insert name into global/local name map
	insertChannelNameMapping(mSubscribedChannelNames, iChannelID, channelID);

	ConcreteChannel<T>* channel;
	// check if already subscribed
	if ( !MIRA_FW.getChannelManager().isSubscribedOn(getGlobalID(), channelID) )
	{
		// set read access rights
		setChannelReadAccess(channelID);

		// If we are a remote authority we will store the information about us
		// subscribing to that channel with the internal flag set to true. Other
		// instances asking the channel manager about non internal subscribers
		// for channels will not see us as a subscriber to avoid cyclic
		// dependencies.
		channel = MIRA_FW.getChannelManager().subscribe<T>(channelID,
		                                                   getGlobalID(),
		                                                   storageDuration,
		                                                   mIsInvisiblePublisherSubscriber);
	}
	else {
		// already subscribed so get channel and test if types match
		channel = MIRA_FW.getChannelManager().getConcreteChannel<T>(channelID);
		// we were subscribed before - with callback?
		bool foundExistingCallback = false;
		for (const auto& i : mSubscriberList) {
			if (i->getChannel() == channel) {
				foundExistingCallback = true;
				break;
			}
		}

		// if not, we want to remove ourselves from the list of 'subscribers without callback'
		// (since we are adding a callback now)
		if (!foundExistingCallback)
			channel->removeSubscriber();
	}

	// default case: use the main thread
	DispatcherThread* dispatcher = &getRuntime()->getMainDispatcher();
	// first alternative: INDEPENDENT_SUBSCRIBER_THREAD - use separate dispatcher
	if (mSubscriberDispatcher)
		dispatcher = mSubscriberDispatcher.get();
	// second alternative: use an own dispatcher for this subscription
	if (independentThread) {
		// independentThread, always creates a new dispatcher
		DispatcherThreadPtr s(new DispatcherThread(getGlobalID()+"_subscriber_"+iChannelID));
		mIndependentSubscriberDispatchers.push_back(s);
		dispatcher = s.get();
		s->start();
	}

	typename ConcreteChannel<T>::SubscriberPtr subscriber(new typename ConcreteChannel<T>::Subscriber(dispatcher, fn, this));
	subscriber->enable(mSubscriberEnabled);
	channel->addSubscriber(subscriber);
	mSubscriberList.push_back(subscriber);
	return toProxy(channel);
}

template<typename T>
inline Channel<T> Authority::subscribe(const std::string& channelID,
                                       boost::function<void (ChannelRead<T>)> fn,
                                       bool independentThread)
{
	return subscribe(channelID, fn, Duration::seconds(0), independentThread);
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribe(const std::string& channelID,
                                       void (Class::*f)(ChannelRead<T>),
                                       Class* obj, 
                                       const Duration& storageDuration,
                                       bool independentThread)
{
	return subscribe<T>(channelID, boost::bind(f, obj, _1),
	                     storageDuration, independentThread );
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribe(const std::string& channelID,
                                       void (Class::*f)(ChannelRead<T>),
                                       Class* obj, 
                                       bool independentThread)
{
	return subscribe<T>(channelID, boost::bind(f, obj, _1), independentThread);
}

template<typename T>
inline Channel<T> Authority::publishAndSubscribe(const std::string& iChannelID,
                                                 const Duration& storageDuration)
{
	publish<T>(iChannelID);
	return subscribe<T>(iChannelID, storageDuration);
}

template<typename T>
inline Channel<T> Authority::publishAndSubscribe(const std::string& iChannelID,
                                                 boost::function<void (ChannelRead<T>)> fn,
                                                 const Duration& storageDuration,
                                                 bool independentThread)
{
	publish<T>(iChannelID);
	return subscribe<T>(iChannelID, fn, storageDuration, independentThread);
}

template<typename T>
inline Channel<T> Authority::publishAndSubscribe(const std::string& channelID,
                                                 boost::function<void (ChannelRead<T>)> fn,
                                                 bool independentThread)
{
	return publishAndSubscribe<T>(channelID, fn, Duration::seconds(0),
	                               independentThread);
}

template<typename T, typename Class>
inline Channel<T> Authority::publishAndSubscribe(const std::string& channelID,
                                                 void (Class::*f)(ChannelRead<T>),
                                                 Class* obj,
                                                 const Duration& storageDuration,
                                                 bool independentThread)
{
	return publishAndSubscribe<T>(channelID, boost::bind(f, obj, _1),
	                               storageDuration, independentThread);
}

template<typename T, typename Class>
inline Channel<T> Authority::publishAndSubscribe(const std::string& channelID,
                                                 void (Class::*f)(ChannelRead<T>),
                                                 Class* obj,
                                                 bool independentThread)
{
	return publishAndSubscribe<T>(channelID, boost::bind(f, obj, _1),
	                               independentThread);
}

///@cond INTERNAL
namespace detail {
/**
 * Internal helper class for subscribing to an interval of data. It provides
 * a callback that can be used by the "normal" subscribe methods above. The
 * callback internally makes sure that no data is missed. Therefore, it stores
 * the timestamp of the last read and tries to read the interval of data from
 * that last timestamp up to the current time. It then delegates the call to
 * the callback provided by the user and passes the interval as parameter.
 */
template <typename T>
class IntervalSubscriber
{
public:
	IntervalSubscriber(const std::string& channelId,
	                   boost::function<void (ChannelReadInterval<T>)> cb,
	                   const Duration& storageDuration,
	                   const Time& startAfter) :
		mChannelId(channelId), mCallback(cb),
		mLast(startAfter), mStorageDuration(storageDuration), mFirstCallback(true) {
	}

	void setChannel(Channel<T> channel) {
		mChannel = channel;
		MIRA_FW.getChannelManager().setStorageDuration(mChannelId, mStorageDuration);
	}

	static void callback(boost::shared_ptr<IntervalSubscriber> self, ChannelRead<T> r)
	{
		assert(self);
		self->internalCallback();
	}

private:

	void internalCallback() {

		ChannelReadInterval<T> interval;

		if (!mFirstCallback)
			interval = mChannel.readInterval(mLast);
		else {
			bool increase = mChannel.isAutoIncreasingStorageDuration();

			// get all slots that are there, without affecting the storage duration
			MIRA_FW.getChannelManager().setAutoIncreaseStorageDuration(mChannelId, false);
			interval = mChannel.readInterval(mLast);

			MIRA_FW.getChannelManager().setAutoIncreaseStorageDuration(mChannelId, increase);
		}

		if(interval.empty())
			return;
		mCallback(interval);
		typename ChannelReadInterval<T>::iterator back = interval.end(); --back;
		mLast = back->timestamp;
		mFirstCallback = false;
	}

private:
	Channel<T> mChannel;
	std::string mChannelId;
	boost::function<void (ChannelReadInterval<T>)> mCallback;
	Time mLast;
	Duration mStorageDuration;
	bool mFirstCallback;
};

class IntervalSubscriberNoLambda
{
public:
	template <typename T>
	static void helpSubscribe(boost::function<void(ChannelRead<T>)> func,
	                          ChannelReadInterval<T> interval)
	{
		typename ChannelReadInterval<T>::iterator it = interval.begin();
		while (it != interval.end()) {
			boost::bind(func, (ChannelRead<T>)it)();
			++it;
		}
	}
};

}
///@endcond INTERNAL

template<typename T>
inline Channel<T> Authority::subscribeInterval(const std::string& channelID,
                     boost::function<void (ChannelReadInterval<T>)> fn,
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	// create helper
	using detail::IntervalSubscriber;
	boost::shared_ptr<IntervalSubscriber<T> > intervalsub(
		new IntervalSubscriber<T>(resolveName(channelID), fn, storageDuration, startAfter));

	// the IntervalSubscriber shared_ptr is binded to the callback, this makes
	// sure, that it will be alive until the subscription is destroyed.
	Channel<T> channel = this->subscribe<T>(channelID,
	    boost::bind(IntervalSubscriber<T>::callback, intervalsub, _1),
	    storageDuration, independentThread);
	intervalsub->setChannel(channel);
	return channel;
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribeInterval(const std::string& channelID,
                     void (Class::*f)(ChannelReadInterval<T>),
                     Class* obj,
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	return subscribeInterval<T>(channelID, boost::bind(f, obj, _1),
	                            storageDuration, independentThread, startAfter);
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribeInterval(const std::string& channelID,
                     void (Class::*f)(ChannelReadInterval<T>),
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::subscribeInterval(..., Class::member, ...)' on an object that is "
	       "not of type Class (you may need to provide a pointer to the correct object)!");
	return subscribeInterval<T>(channelID, f, obj, storageDuration, independentThread, startAfter);
}

template<typename T>
inline Channel<T> Authority::subscribeIntervalByElements(const std::string& channelID,
                     boost::function<void(ChannelRead<T>)> func,
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	// Create wrapper function to iterate through interval
	boost::function<void (ChannelReadInterval<T>)> function =
			boost::bind(&detail::IntervalSubscriberNoLambda::helpSubscribe<T>, func, _1);

	return subscribeInterval<T>(channelID, function, storageDuration,
	                            independentThread, startAfter);
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribeIntervalByElements(const std::string& channelID,
                     void (Class::*f)(ChannelRead<T>),
                     Class* obj,
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	// Bind object to member function
	boost::function<void(ChannelRead<T>)> func = boost::bind(f, obj, _1);

	return subscribeIntervalByElements<T>(channelID, func, storageDuration,
	                                      independentThread, startAfter);
}

template<typename T, typename Class>
inline Channel<T> Authority::subscribeIntervalByElements(const std::string& channelID,
                     void (Class::*f)(ChannelRead<T>),
                     const Duration& storageDuration,
                     bool independentThread,
                     const Time& startAfter)
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::subscribeIntervalByElements(..., Class::member, ...)' on an object "
	       "that is not of type Class (you may need to provide a pointer to the correct object)!");
	return subscribeIntervalByElements<T,Class>(channelID, f, obj, storageDuration,
	                                            independentThread, startAfter);
}



template <typename T>
inline ChannelRead<T> Authority::waitForData(Channel<T> channel,
                                             Duration timeout) const
{
	return channel.waitForData(timeout);
}

inline void Authority::unsubscribe(const std::string& iChannelID)
{
	validate();
	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	eraseChannelNameMapping(mSubscribedChannelNames, iChannelID, channelID);

	// check if really subscribed
	if ( !MIRA_FW.getChannelManager().isSubscribedOn(getGlobalID(), channelID) )
		return;

	// we not longer have read access rights
	removeChannelReadAccess(channelID);

	MIRA_FW.getChannelManager().removeSubscriber(channelID, getGlobalID());

	bool removedCallback = false;

	// Check if we have subscribed with a callback - find the subscriber ptr
	SubscribePtrList::iterator i = mSubscriberList.begin();
	while ( i!= mSubscriberList.end()) {
		if ( (*i)->getChannel()->getID() == channelID )
		{
			// we have subscribed with a callback so
			// -if subscribed with independent thread destroy and stop the handler
			// -remove the subscriber
			AbstractChannelSubscriberPtr subscriber = *i;
			subscriber->enable(false);
			// make sure the all signaled callbacks to this subscriber are done/removed
			subscriber->unsignal();
			// check if we subscribed with independent thread
			if (subscriber->getDispatcher() != &getRuntime()->getMainDispatcher())
			{
				IndependentSubscribers::iterator h = mIndependentSubscriberDispatchers.begin();
				for (; h!= mIndependentSubscriberDispatchers.end(); ++h)
					if(h->get() == subscriber->getDispatcher())
						break;

				if ( h != mIndependentSubscriberDispatchers.end() )
				{
					(*h)->stop();
					mIndependentSubscriberDispatchers.erase(h);
				}
			}
			subscriber->getChannel()->removeSubscriber(subscriber);
			SubscribePtrList::iterator it = i++;
			mSubscriberList.erase(it);
			removedCallback = true;
		}
		else
			++i;
	}
	if (!removedCallback)
		MIRA_FW.getChannelManager().deductNumSubscribers(channelID);
}

template <typename T>
inline void Authority::unsubscribe(const std::string& iChannelID)
{
	unsubscribe(iChannelID);
}

template <typename T>
inline Channel<T> Authority::getChannel(const std::string& iChannelID)
{
	validate();
	// resolve channel's fully qualified name
	std::string channelID = resolveName(iChannelID);
	ConcreteChannel<T>* channel = MIRA_FW.getChannelManager().getConcreteChannel<T>(channelID);
	return toProxy(channel);
}

template<typename T>
inline Channel<T> Authority::toProxy(ConcreteChannel<T>* channel)
{
	// return channel proxy for channel
	return Channel<T>(channel, mAccessMap[channel->getID()].flags);
}


namespace detail {

template<typename Transform, typename Filter>
class TransformSubscriber
{
public:

	TransformSubscriber(FrameworkTransformerNode* t, FrameworkTransformerNode* s,
	                    Filter&& f, boost::function<void (Transform, Time)> fn) :
	                    target(t), source(s), filter(f), callback(fn) {}

	static void onFrameChanged(ChannelRead<void> tf, boost::shared_ptr<TransformSubscriber> self)
	{
		Time time = tf.getTimestamp();
		self->callback(MIRA_FW.getTransformer()->getTransform<Transform>
		                      (self->target, self->source, time, self->filter), time);
	}

public:
	FrameworkTransformerNode* target;
	FrameworkTransformerNode* source;
	Filter filter;
	boost::function<void (Transform, Time)> callback;
};

}

template<typename Transform, typename Filter>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          boost::function<void (Transform, Time)> fn,
                                          Filter&& filter)
{
	FrameworkTransformerNode* target = getTransformNode(targetID);
	FrameworkTransformerNode* source = getTransformNode(sourceID);

	TransformerBase::Chain chain;
	MIRA_FW.getTransformer()->getTransformChain(target, source, chain); // throws XTransform


	typedef detail::TransformSubscriber<Transform,Filter> Subscriber;
	boost::shared_ptr<Subscriber> subscriberPtr(new Subscriber(target,source,std::move(filter),fn));

	// binding the subscriberPtr to the callback ensures, that the
	// TransformSubscriber helper object keeps alive as long as the callback exists
	foreach(const TransformerBase::AbstractNodePtr& node, chain.inverse)
		subscribe<void>(node->getID(), boost::bind(Subscriber::onFrameChanged, _1, subscriberPtr));

	foreach(const TransformerBase::AbstractNodePtr& node, chain.forward)
		subscribe<void>(node->getID(), boost::bind(Subscriber::onFrameChanged, _1, subscriberPtr));
}

template<typename Transform, typename Filter, typename Class>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          void (Class::*f)(Transform, Time), Class* obj,
                                          Filter&& filter)
{
	subscribeTransform<Transform,Filter>(targetID, sourceID, boost::bind(f, obj, _1, _2), filter);
}

template<typename Transform, typename Filter, typename Class>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          void (Class::*f)(Transform, Time),
                                           Filter&& filter)
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::subscribeTransform(..., Class::member, ...)' on an object "
	       "that is not of type Class (you may need to provide a pointer to the correct object)!");
	subscribeTransform<Transform,Filter,Class>(targetID, sourceID, f, obj, filter);
}

template<typename Transform>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          boost::function<void (Transform, Time)> fn)
{
	subscribeTransform<Transform,NearestNeighborInterpolator>(targetID, sourceID, fn, NearestNeighborInterpolator());
}

template<typename Transform, typename Class>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          void (Class::*f)(Transform, Time), Class* obj)
{
	subscribeTransform<Transform>(targetID, sourceID, boost::bind(f, obj, _1, _2));
}

template<typename Transform, typename Class>
inline void Authority::subscribeTransform(const std::string& targetID,
                                          const std::string& sourceID,
                                          void (Class::*f)(Transform, Time))
{
	Class* obj = dynamic_cast<Class*>(this);
	assert(obj!=NULL &&
	       "Called 'Authority::subscribeTransform(..., Class::member, ...)' on an object "
	       "that is not of type Class (you may need to provide a pointer to the correct object)!");
	subscribeTransform<Transform,Class>(targetID, sourceID, f, obj);
}



template<typename Transform, typename Filter>
inline Transform Authority::getTransform(const std::string& targetID,
                                         const std::string& sourceID,
                                         const Time& time, Filter&& filter) const
{
	FrameworkTransformerNode* target = getTransformNode(targetID);
	FrameworkTransformerNode* source = getTransformNode(sourceID);
	return MIRA_FW.getTransformer()->getTransform<Transform>(target, source, time, filter);
}


template<typename Transform, typename Filter>
inline Transform Authority::getTransform(const std::string& targetID, const Time& targetTime,
                                         const std::string& sourceID, const Time& sourceTime,
                                         const std::string& fixedID,
                                         Filter&& filter) const
{
	FrameworkTransformerNode* target = getTransformNode(targetID);
	FrameworkTransformerNode* source = getTransformNode(sourceID);
	FrameworkTransformerNode* fixed  = getTransformNode(fixedID);
	return MIRA_FW.getTransformer()->getTransform<Transform>(target, targetTime,
	                                                         source, sourceTime,
	                                                         fixed, filter);
}


inline FrameworkTransformerNode* Authority::getTransformNode(const std::string& frameID) const
{
	std::string id = resolveName(frameID);
	FrameworkTransformerNode* node = MIRA_FW.getTransformer()->getNode(id);
	if(node==NULL)
		MIRA_THROW(XInvalidRead, "Frame '" << id << "' does not exist");
	return node;

}

inline TransformDesc Authority::prepareTransform(const std::string& targetID,
                                                 const std::string& sourceID) const
{
	FrameworkTransformerNode* target = getTransformNode(targetID);
	FrameworkTransformerNode* source = getTransformNode(sourceID);
	return MIRA_FW.getTransformer()->prepareTransform(target, source);
}

inline TransformDesc Authority::prepareTransform(const std::string& targetID,
                                                 const std::string& sourceID,
                                                 const std::string& fixedID) const
{
	FrameworkTransformerNode* target = getTransformNode(targetID);
	FrameworkTransformerNode* source = getTransformNode(sourceID);
	FrameworkTransformerNode* fixed = getTransformNode(fixedID);
	return MIRA_FW.getTransformer()->prepareTransform(target, source, fixed);
}

template<typename Transform, typename Filter>
inline Transform Authority::getTransform(const TransformDesc& desc,
                                         const Time& time, Filter&& filter) const
{
	return MIRA_FW.getTransformer()->getTransform<Transform>(desc, time, filter);
}

template<typename Transform, typename Filter>
inline Transform Authority::getTransform(const TransformDesc& desc,
                                         const Time& targetTime,
                                         const Time& sourceTime, Filter&& filter) const
{
	return MIRA_FW.getTransformer()->getTransform<Transform>(desc, targetTime,
	                                                         sourceTime, filter);
}

template<typename Transform>
inline void Authority::publishTransform(const std::string& frameID,
                                        const Transform& transform,
                                        const Time& time) const
{
	std::string id = resolveName(frameID);
	MIRA_FW.getTransformer()->publishTransform(id, transform, time);
}

template<typename Transform, typename Filter>
inline void Authority::publishTransformIndirect(const std::string& frameID,
                                                const std::string& targetID,
                                                const std::string& sourceID,
                                                const Transform& transform,
                                                const Time& time, Filter&& filter) const
{
	std::string frame  = resolveName(frameID);
	std::string target = resolveName(targetID);
	std::string source = resolveName(sourceID);
	MIRA_FW.getTransformer()->publishTransformIndirect(frame, target, source,
	                                                   transform, time, filter);
}

inline void Authority::addTransformLink(const std::string& childID,
                                        const std::string& parentID) const
{
	std::string child = resolveName(childID);
	std::string parent = resolveName(parentID);
	MIRA_FW.getTransformer()->addLink(child, parent);
}

inline void Authority::addTransformLink(const std::string& childID,
                                        const std::string& parentID,
                                        FrameworkTransformerNode::Type type) const
{
	std::string child = resolveName(childID);
	std::string parent = resolveName(parentID);
	MIRA_FW.getTransformer()->addLink(child, parent, type);
}

template<typename Service>
inline void Authority::publishService(const std::string& name, Service& service,
                                      RPCHandlerPtr handler)
{
	validate();
	if (mIsInternal)
		MIRA_THROW(XRuntime, "Internal authorities can not have services");

	std::string resolvedname = resolveName(name);

	if (!handler)
		handler = mRPCHandler;
	MIRA_FW.getRPCManager().addLocalService(resolvedname, service,
	                                        handler);
	remotePublishService(resolvedname);
	mPublishedServices.insert(resolvedname);
}

template<typename Service>
inline void Authority::publishService(Service& service, RPCHandlerPtr handler)
{
	publishService(this->getGlobalID(), service, handler);
}

template<typename Class>
inline void Authority::registerCallbackForInterface(const std::string& interface,
                                                    void (Class::*f)(const std::string&,
                                                                     const std::string&),
                                                    Class* obj)
{
	registerCallbackForInterface(interface, boost::bind(f, obj, _1, _2));
}

///////////////////////////////////////////////////////////////////////////////

template<typename R, typename... ARGS>
inline RPCFuture<R> Authority::callService(const std::string& iService, const std::string& method,
                                           ARGS&&... args) const
{
	validate();
	std::string service = resolveServiceName(iService);
	return MIRA_FW.getRPCManager().call<R>(std::move(service), method, std::forward<ARGS>(args)...);
}

template<class F>
ServiceCall<F> Authority::createServiceCall(const std::string& service, std::string method,
                                            bool waitTillExists, bool showBootupMsg)
{
	validate();
	auto serviceProvider = resolveServiceName(service);
	auto res = ServiceCall<F>{serviceProvider, std::move(method)};

	if (waitTillExists) {
		if (showBootupMsg) {
			bootup(MakeString() << "Waiting for service method " << serviceProvider
			                    << "::" << res.getSignature());
		}
		res.waitForServiceCall();
	}
	return res;
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
