/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeVisitor.h
 *    Class implementing a visitor concept for tapes.
 *
 * @author Tim Langner
 * @date   2010/12/29
 */

#ifndef _MIRA_TAPEVISITOR_H_
#define _MIRA_TAPEVISITOR_H_

#include <fw/Tape.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * Visitor class to inspect tapes by channels and/or time interval.
 * All times of messages are relative times to the start time of the visitor.
 * The start time will be the smallest time of all visited tapes.
 * Multiple tapes can be inspected at once.
 * Their times will be merged relative to the smallest time in all tapes.
 * This means if two tapes are recorded with a ten minute delay
 * this delay is also represented in the visitor.
 */
class TapeVisitor
{
public:

	/// Instance of a message from a tape
	struct MessageInstance
	{
		MessageInstance() :
			tape(NULL), offset(0) {}

		/// pointer to tape the message is stored in
		Tape* tape;

		/// offset of the message in the tape
		uint64 offset;

		/// frame id of the channel
		std::string frameID;

		/// sequence id
		uint32 sequenceID;

		/// serialized message data
		Buffer<uint8> data;

		/// specifies whether the data was compressed within the tape or not
		bool compressed;
	};

	/// Info about a message entry in a tape
	struct MessageInfo
	{
		MessageInfo() :
			tape(NULL),
			info(NULL),
			index(NULL) {}

		/// pointer to tape the message is stored in
		Tape* tape;

		/// time offset of the message relative to start of recording
		Duration timeOffset;

		/// pointer to the info object of the channel
		const Tape::ChannelInfo* info;

		/// pointer to the index object of the message
		const Tape::MessageIndex* index;

		bool operator<(const MessageInfo& other) const
		{
			return timeOffset < other.timeOffset;
		}
	};

	/// Vector of message informations
	typedef std::vector<MessageInfo> MessageMap;
	/// Maps channel info to channel names
	typedef std::map<std::string, const Tape::ChannelInfo*> ChannelMap;

	/// STL conform iterator
	class iterator;
	friend class iterator;

	/**
	 * Iterator to iterate over all messages in all tapes visited by the visitor
	 */
	class iterator
	{
	public:
		iterator() :
			mVisitor(NULL) {}

		iterator(TapeVisitor* visitor, MessageMap::iterator message) :
			mVisitor(visitor),
			mMessage(message) {}

		iterator& operator=(const iterator& other)
		{
			mVisitor = other.mVisitor;
			mMessage = other.mMessage;
			return *this;
		}

		iterator& operator++()
		{
			++mMessage;
			return *this;
		}

		iterator& operator--()
		{
			--mMessage;
			return *this;
		}

		/**
		 * Instantiates the message by loading it from the underlying tape file.
		 */
		const MessageInstance& operator*() const
		{
			if (!mVisitor || mMessage == mVisitor->mMessages.end() )
				MIRA_THROW(XIO, "Cannot instantiate message from the end iterator");

			// If we are already instantiated return us
			if (mInstance.tape == mMessage->tape &&
				mInstance.offset == (mMessage->index->block+mMessage->index->offset))
				return mInstance;

			// copy informations about the message
			mInstance.tape = mMessage->tape;
			mInstance.offset = mMessage->index->block+mMessage->index->offset;

			// read the message from tape
			Duration oTime;
			mMessage->tape->readMessage(*mMessage->index, mInstance.frameID,
			                            mInstance.sequenceID, mInstance.data, oTime,
			                            mInstance.compressed);
			return mInstance;
		}

		/**
		 * Instantiates the message by loading it from the underlying tape file.
		 */
		const MessageInstance* operator->() const
		{
			return &this->operator *();
		}

		/**
		 * Instantiates the message by loading it from the underlying tape file
		 * and deserializes its content into oData.
		 * This method is provided for convenience.
		 *
		 * @throw XIO If the type of the message does not match T
		 */
		template <typename T>
		void read(Stamped<T>& oData)
		{
			validate();
			// perform type check
			if(mMessage->info->type!=typeName<T>()) {
				MIRA_THROW(XIO, "Cannot read a '"<<typeName<T>()<<
					"' from Tape-Iterator message which has type '"<<mMessage->info->type<<"'");
			}

			// instantiate the data
			const MessageInstance& msgInst = this->operator*();
			Buffer<uint8> buf(msgInst.data);
			BinaryBufferDeserializer bs(&buf);
			// deserialize it ...
			bs.deserialize(oData.value(),false);
			// and fill in the header info
			oData.timestamp = mVisitor->getStartTime() + getTimeOffset();
			oData.frameID = msgInst.frameID;
			oData.sequenceID = msgInst.sequenceID;
		}

		/**
		 * Instantiates the message by loading it from the underlying tape file
		 * and deserializes its content as JSONValue into oData.
		 * This method is provided for convenience.
		 */
		void readJSON(Stamped<JSONValue>& oData);


		bool operator==(const iterator& other) const
		{
			return mVisitor == other.mVisitor && mMessage == other.mMessage;
		}

		bool operator!=(const iterator& other) const
		{
			return mVisitor != other.mVisitor || mMessage != other.mMessage;
		}

		/**
		 * Get the offset of the recording time for this message.
		 * Note if the message is not loaded from file calling this function
		 * will NOT load it. Calling this is fast!
		 */
		Duration getTimeOffset() const
		{
			validate();
			return mMessage->timeOffset;
		}

		/**
		 * Get the information object about the channel this
		 * message belongs to.
		 * Note if the message is not loaded from file calling this function
		 * will NOT load it. Calling this is fast!
		 */
		const Tape::ChannelInfo* getChannelInfo() const {
			validate();
			return mMessage->info;
		}

		/**
		 * Get the time zone offset of the tape containing this message.
		 */
		Duration getTZOffset() const
		{
			validate();
			return mMessage->tape->getTZOffset();
		}


		friend class TapeVisitor;

	protected:

		void validate() const
		{
			if (!mVisitor || mMessage == mVisitor->mMessages.end())
				MIRA_THROW(XIO, "Cannot access message at the end iterator");
		}

		mutable MessageInstance mInstance;
		TapeVisitor* mVisitor;
		MessageMap::iterator mMessage;
	};

	/**
	 * Visit a tape by specifying an interval of the relative time of the first
	 * and the last message.
	 * If a start time has been set (either by previous calls to visit(), or by setStartTime()),
	 * the default is to adapt the start time to the minimum of the previous start time and this
	 * tape's start time. All added messages are then reindexed to adapt their relative time.
	 * @param[in] iTape The tape to visit
	 * @param[in] first The first offset relative to the start of recording
	 *            (Where to start our visit)
	 * @param[in] last The last offset relative to the start of recording
	 *            (Where to end our visit)
	 * @param[in] ignoreStartTime Ignore the start time of this tape when re-indexing.
	 *                            Ignoring ALL visited tapes' start times and never setting any
	 *                            common start time will result in invalid start time and messages
	 *                            from different tapes not being ordered properly.
	 */
	void visit(Tape* iTape,
	           Duration first = Duration::negativeInfinity(),
	           Duration last = Duration::infinity(),
	           bool ignoreStartTime = false);

	/**
	 * Same as above, but additionally select which channels to visit.
	 * @param[in] iTape The tape to visit
	 * @param[in] channels The list of channels to visit
	 * @param[in] first The first offset relative to the start of recording
	 *            (Where to start our visit)
	 * @param[in] last The last offset relative to the start of recording
	 *            (Where to end our visit)
	 * @param[in] ignoreStartTime Ignore the start time of this tape when re-indexing. See above.
	 */
	void visit(Tape* iTape, const std::vector<std::string>& channels,
	           Duration first = Duration::negativeInfinity(),
	           Duration last = Duration::infinity(),
	           bool ignoreStartTime = false);

	/**
	 * Get the common start time of the visit for all tapes
	 * @return Start time or invalid time if no tapes visited
	 */
	Time getStartTime() const
	{
		if ( mStart )
			return *mStart;
		return Time::invalid();
	}

	/**
	 * Set the common start time of the visit for all tapes.
	 */
	void setStartTime(const Time& start)
	{
		mStart.reset(start);
		for (uint32 i=0; i<mMessages.size(); ++i)
			mMessages[i].timeOffset = (mMessages[i].tape->getStartTime() +
			                           mMessages[i].index->timeOffset) - *mStart;
	}

	/**
	 * Get the time offset of the first visited message
	 * @return first message time or 0 if no tapes visited
	 */
	Duration getFirstMessageTimeOffset() const
	{
		if (mMessages.size() > 0)
			return mMessages.begin()->timeOffset;
		return Duration::nanoseconds(0);
	}

	/**
	 * Get the time offset of the last visited message
	 * @return last message time or 0 if no tapes visited
	 */
	Duration getLastMessageTimeOffset() const
	{
		if (mMessages.size() > 0)
			return mMessages.rbegin()->timeOffset;
		return Duration::nanoseconds(0);
	}

	/**
	 * Get the time zone offset of the last visited message
	 * @return last message timezone offset or 0 if no tapes visited
	 */
	Duration getLastMessageTZOffset() const
	{
		if (mMessages.size() > 0)
			return mMessages.rbegin()->tape->getTZOffset();
		return Duration::nanoseconds(0);
	}

	/**
	 * Get the number of messages to visit
	 * @return Number of messages
	 */
	uint32 getMessageCount() const
	{
		return mMessages.size();
	}

	/**
	 * Iterator to the first visited message in a tape
	 * @return Iterator to the first message to visit
	 */
	iterator begin()
	{
		return iterator(this, mMessages.begin());
	}

	/**
	 * Iterator to the message at the given offset since start of tape
	 * @param[in] timeOffset The offset of the message
	 * @return Iterator to the message closest to the given time offset
	 */
	iterator at(Duration timeOffset)
	{
		if ( mMessages.size() == 0 )
			return iterator(this, mMessages.end());
		if ( mMessages.size() == 1 )
			return iterator(this, mMessages.begin());
		MessageMap::iterator it = mMessages.begin();
		Duration diffNext = abs(mMessages[0].timeOffset-timeOffset);
		for(uint32 i=0; i<mMessages.size()-1;++i, ++it)
		{
			Duration diffNow = diffNext;
			diffNext = abs(mMessages[i+1].timeOffset-timeOffset);
			if ( diffNow < diffNext )
				return iterator(this, it);
		}
		return iterator(this, mMessages.end());
	}

	/**
	 * Iterator to the end of visited messages
	 * @return Iterator to the end of the messages
	 */
	iterator end()
	{
		return iterator(this, mMessages.end());
	}

	/**
	 * Returns the iterator to the next message from given channel with a
	 * time offset of at least startOffset, or end iterator if no more messages
	 * of that channel exist in the tape.
	 */
	iterator findNextOf(Duration startOffset, const std::string& channel)
	{
		return findNextOf(at(startOffset), channel);
	}

	/**
	 * Returns the iterator to the next message from given channel, starting at
	 * the given start iterator, or end iterator if no more messages of that
	 * channel exist in the tape.
	 */
	iterator findNextOf(const iterator& start, const std::string& channel)
	{
		MessageMap::iterator i = start.mMessage;
		for(; i!=mMessages.end(); ++i)
			if (i->info->name == channel)
				return iterator(this, i);
		return iterator(this, mMessages.end());
	}

	/**
	 * Returns the iterator to the next message from any of the given channels
	 * with a time offset of at least startOffset, or end iterator if no more
	 * messages of those channels exist in the tape.
	 */
	iterator findNextOfAnyOf(Duration startOffset, const std::vector<std::string>& channels)
 	{
		return findNextOfAnyOf(at(startOffset), channels);
	}

	/**
	 * Returns the iterator to the next message from any of the given channels,
	 * starting at the given start iterator, or end iterator if no more messages
	 * of those channels exist in the tape.
	 */
 	iterator findNextOfAnyOf(const iterator& start, const std::vector<std::string>& channels)
	{
		MessageMap::iterator i = start.mMessage;
		for(; i!=mMessages.end(); ++i)
			if (std::find(channels.begin(), channels.end(), i->info->name) != channels.end())
				return iterator(this, i);

		return iterator(this, mMessages.end());
	}

	/**
	 * Get a list of the visited channels
	 * @return The visited channels
	 */
	const ChannelMap& getChannels() const
	{
		return mChannels;
	}

protected:

	void addChannel(Tape* tape, const Tape::ChannelInfo* channel,
	                Duration first = Duration::nanoseconds(0),
	                Duration last = Duration::infinity());

	void reIndex(Tape* tape, bool ignoreStartTime = false);

	boost::optional<Time> mStart;
	MessageMap mMessages;
	ChannelMap mChannels;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
