/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Status.h
 *    Status and status management classes used for diagnostics.
 *
 * @author Tim Langner
 * @date   2011/01/29
 */

#ifndef _MIRA_STATUS_H_
#define _MIRA_STATUS_H_

#include <sstream>
#include <list>

#ifndef Q_MOC_RUN
#include <boost/optional.hpp>
#endif

#include <utils/Time.h>
#include <serialization/Serialization.h>
#include <serialization/adapters/std/map>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Status entry class.
 * Used to signal the current status of a module (OK, WARNING, ERROR)
 */
class Status
{
public:
	/// Different status levels sorted from "okay" to "most severe".
	enum StatusMode
	{
		OK,
		BOOTUP,
		RECOVER,
		WARNING,
		ERROR,
	};

public:

	Status() :
		mode(OK)
	{}

	/// Construct a new status with given mode, category and text.
	Status(StatusMode iMode, const std::string& iCategory,
	       const std::string& iTrText, const std::string& iMessage) :
		mode(iMode),
		category(iCategory),
		trText(iTrText),
		message(iMessage)
	{}

	bool operator==(const Status& other) const
	{
		return mode == other.mode &&
			category == other.category &&
			trText == other.trText &&
			message == other.message;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Mode", mode, "Status mode");
		r.member("Category", category, "Category");
		r.member("TranslatedText", trText, "Translated text");
		r.member("Message", message, "Message");
	}

public:

	/// the status flag (OK, WARNING or ERROR)
	StatusMode mode;

	/// category of that status
	std::string category;

	/// Status text that can be used for translation of the error to other languages
	std::string trText;

	/// the corresponding message
	std::string message;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for modules that want to use diagnostics and set the current status.
 * Classes inheriting this class can be added to the StatusManager for obtaining
 * an overall state of one or multiple diagnostic modules.
 */
class DiagnosticsModule
{
public:

	typedef std::map<std::string, Status> StatusMap;

	DiagnosticsModule(const std::string& name="") :
		mHeartbeatInterval(Duration::seconds(1)),
		mName(name) {}

	virtual ~DiagnosticsModule() {}

	/**
	* Set the watchdog interval.
	*/
	void setHeartbeatInterval(const Duration& interval)
	{
		mHeartbeatInterval = interval;
	}

	void setName(const std::string& name)
	{
		mName = name;
	}

	/**
	 * Return the watchdog interval.
	 */
	Duration getHeartbeatInterval() const
	{
		return mHeartbeatInterval;
	}

	/**
	 * When called for the first time heartbeat usage will be enabled.
	 * This resets the watchdog and signal that everything is working.
	 * If you call heartbeat in an interval < watchdog interval the status
	 * will be OK (if no other error is set)
	 * If a module blocks or fails and does not call heartbeat again until
	 * the watchdog interval passed the status will be ERROR
	 */
	void heartbeat()
	{
		mLastHeartbeat.reset(Time::now());
	}

	/**
	 * Returns true if heartbeat usage is enabled (by first call to heartbeat())
	 * and last heartbeat was more than heartbeat interval time ago.
	 */
	bool hasHeartbeatTimeout() const
	{
		return mLastHeartbeat && (Time::now()-*mLastHeartbeat) > mHeartbeatInterval;
	}

	/**
	 * Signal that the module is booting up. Optional a message text and
	 * a text used for translation can be specified
	 */
	void bootup(const std::string& message, const std::string& trText="");

	/**
	 * Signal that the module has finished booting up (There can still be
	 * errors but booting up is finished)
	 */
	void bootupFinished();

	/**
	 * Signal that the module has finished recovering (There can still be
	 * errors but recovering is finished)
	 */
	void recoverFinished();

	/**
	 * Signal that the module is recovering. Optional a message text and
	 * a text used for translation can be specified
	 */
	void recover(const std::string& message, const std::string& trText="");

	/**
	 * Signal that a category contains no more errors.
	 * If no category is set all categories are cleared (except boot up).
	 */
	void ok(const std::string& category="");

	/**
	 * Signal a warning in a category.
	 * A message and an optional text for translation can be given.
	 * Additionally, the message is written to the error log.
	 * @return true if warning was set, false if a warning with same text was already set
	 */
	bool warning(const std::string& category, const std::string& message,
	             const std::string& trText="");

	/**
	 * Signal an error in a category.
	 * A message and an optional text for translation can be given.
	 * Additionally, the error message is written to the error log.
	 * @return true if error was set, false if a error with same text was already set
	 */
	bool error(const std::string& category, const std::string& message,
	           const std::string& trText="");

	/**
	 * Set the status of a given category.
	 * This method can be used to set the warning or error status of the
	 * specified category. a message and an optional text for translation
	 * can be given.
	 * This method essentially does the same as the above warning() and error()
	 * methods. In contrast to those two methods, this method does NOT write
	 * the error message to the error/warning log. Hence, it can be used to
	 * set the error and warning status silently. In all other cases the
	 * warning() and error() methods should be used, which are more convenient.
	 *
	 * @return true if error was set, false if a error with same text was already set
	 */
	bool setStatus(Status::StatusMode mode, const std::string& category,
	               const std::string& message, const std::string& trText="");

	/**
	 * Get the status mode defined by the status of the watchdog (if used) and
	 * the bootup, error and warning methods.
	 * If the watchdog is used (watchdog is enabled when calling heartbeat()
	 * for the first time) this method tests if the latest reset of the
	 * watchdog was inside the specified watchdog interval.
	 * If not it will return ERROR.
	 */
	Status::StatusMode getStatus() const;

	/**
	 * Gets the current status map containing all errors, warnings and
	 * bootup messages.
	 */
	StatusMap getStatusMap() const;

protected:

	boost::optional<Status> mBootUpStatus;
	boost::optional<Status> mRecoverStatus;
	Duration mHeartbeatInterval;
	boost::optional<Time> mLastHeartbeat;
	StatusMap mStatusMap;
	std::string mName;
};

/// Typedef of a DiagnosticsModule pointer.
typedef DiagnosticsModule* DiagnosticsModulePtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Manages the status of one or multiple modules inheriting from DiagnosticsModule.
 */
class StatusManager
{
public:

	typedef std::multimap<std::string, Status> StatusMap;

	/// Return the overall (most severe) status from a range of status map entries
	static Status::StatusMode getOverallStatus(StatusMap::const_iterator start,
	                                           StatusMap::const_iterator end)
	{
		Status::StatusMode mostSevereMode = Status::OK;
		for(auto i = start; i != end; ++i)
		{
			if (mostSevereMode < i->second.mode)
				mostSevereMode = i->second.mode;
		}
		return mostSevereMode;
	}

	/// Set the id that is used for displaying status messages
	void setID(const std::string& id)
	{
		mID = id;
	}

	/**
	 * Get the overall status mode defined by the status of all registered
	 * diagnostic modules.
	 */
	Status::StatusMode getStatus() const;

	/**
	 * Get the status of the specified diagnostic module.
	 */
	Status::StatusMode getStatus(const std::string& diagnosticModule) const;

	/**
	 * Register a new diagnostics module by specifying a name and a pointer to
	 * the module.
	 * The status of this module is taken into account when getting the overall
	 * status using @ref getStatus().
	 */
	void registerDiagnostics(const std::string& name, DiagnosticsModulePtr ptr);

	/**
	 * Unregister a diagnostics module. The status of this module will not
	 * longer contribute to the overall status.
	 */
	void unregisterDiagnostics(const std::string& name);

	/**
	 * Unregister a diagnostics module. The status of this module will not
	 * longer contribute to the overall status.
	 */
	void unregisterDiagnostics(DiagnosticsModulePtr ptr);

	/**
	 * Get the list of all registered modules
	 */
	std::vector<std::string> getDiagnosticModules() const;

	/**
	 * Gets the current status map
	 */
	StatusMap getStatusMap() const;

	/**
	 * Gets the current status map of the specified diagnostics module
	 */
	DiagnosticsModule::StatusMap getStatusMap(const std::string& diagnosticModule) const;

	template <typename Reflector>
	void reflect(Reflector& r)
	{
	} // do nothing here

private:
	typedef std::map<std::string, DiagnosticsModulePtr> ModuleMap;
	// vector used for storing the order the modules were registered
	std::vector<std::string> mModuleOrder;
	mutable boost::mutex mModuleMutex;
	ModuleMap mModules;
	StatusMap mStatus;
	std::string mID;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
