/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Runnable.h
 *    TODO Add description.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   14/04/16
 */

#ifndef _MIRA_RUNNABLE_H_
#define _MIRA_RUNNABLE_H_

#ifndef Q_MOC_RUN
#include <boost/enable_shared_from_this.hpp>
#endif

#include <error/LoggingAux.h>
#include <fw/Status.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

//fwd decls.
class IRunnable;
class DispatcherThread;

/// a runnable pointer
typedef boost::shared_ptr<IRunnable> IRunnablePtr;

/**
 * Base for all runnable classes that can be signaled when they need to
 * run again.
 */
class IRunnable : public boost::enable_shared_from_this<IRunnable>
{
public:

	virtual ~IRunnable() {}

protected:
	friend class DispatcherThread;

	/**
	 * Called from dispatcher thread this runnable is attached to,
	 * whenever the runnable should be executed.
	 */
	virtual void run(DispatcherThread* dispatcher) = 0;

};

///////////////////////////////////////////////////////////////////////////////

class DiagnosticRunnable : public IRunnable
{
public:

	DiagnosticRunnable(DiagnosticsModulePtr module = NULL) :
		mDiagnosticsModule(module) {}

	void setDiagnosticsModule(DiagnosticsModulePtr module) {
		mDiagnosticsModule = module;
	}
	DiagnosticsModulePtr getDiagnosticsModule() { return mDiagnosticsModule; }

protected:

	DiagnosticsModulePtr mDiagnosticsModule;
};
typedef boost::shared_ptr<DiagnosticRunnable> DiagnosticRunnablePtr;

///////////////////////////////////////////////////////////////////////////////

/**
 * Runnable that wraps any function pointer.
 */
template <typename F>
class FunctionRunnable : public DiagnosticRunnable
{
public:
	FunctionRunnable(F&& fn, DiagnosticsModulePtr module = NULL) :
		DiagnosticRunnable(module), function(fn) {}

	virtual void run(DispatcherThread* dispatcher) {

		try {
			function();
		}
		catch(Exception& ex) {
			if (!mDiagnosticsModule ||
				mDiagnosticsModule->error("Runnable", ex.message(),
				                         "Exception while invoking Runnable")) {
				MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
			}
		}
		catch(std::exception& ex) {
			if (!mDiagnosticsModule ||
				mDiagnosticsModule->error("Runnable", ex.what(),
				                         "Exception while invoking Runnable")) {
				MIRA_LOG_EXCEPTION(ERROR,ex) << "Exception:\n";
			}
		}
	}

	F function;

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
