/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRAAdapter.h
 *    An example for bringing ROS and MIRA together.
 *
 * @author Tim Langner, Christian Reuther
 * @date   2012/03/20
 */

#include "RobotDataTypesConversion.h"

///////////////////////////////////////////////////////////////////////////////

class Converter
{
public:
	Converter() : authority("/", "miraadapter", mira::Authority::ANONYMOUS) {
		// Empty
	}

	template <typename T>
	void rosCallback(const typename MIRA2ROSTrait<T>::type::ConstPtr& msg,
	                 const std::string& channel)
	{
		mira::Channel<T> c = authority.getChannel<T>(channel);
		mira::ChannelWrite<T> w = c.write();
		ros2mira(msg, w);
	}

	template <typename T>
	void miraCallback(mira::ChannelRead<T> data, const std::string& channel)
	{
		typename MIRA2ROSTrait<T>::type msg;
		mira2ros(data, msg);
		rosPublisher[channel].publish(msg);
	}

	template <typename T>
	void convertROS2MIRA(const std::string& topic, const std::string& channel)
	{
		ROS_INFO("Adding conversion from topic %s to channel %s",
		         topic.c_str(), channel.c_str());
		authority.publish<T>(channel);
		rosSubscriber[channel] = node.subscribe<typename MIRA2ROSTrait<T>::type>(topic, 1000, 
		                                                                         boost::bind(&Converter::rosCallback<T>, this, _1, channel));
	}

	template <typename T>
	void convertMIRA2ROS(const std::string& channel, const std::string& topic)
	{
		ROS_INFO("Adding conversion from channel %s to topic %s", 
		         channel.c_str(), topic.c_str());
		rosPublisher[channel] = node.advertise<typename MIRA2ROSTrait<T>::type>(topic, 1000);
		authority.subscribe<T>(channel,
		                       boost::bind(&Converter::miraCallback<T>, this, _1, channel));
	}

	mira::Authority authority;
	ros::NodeHandle node;
	std::map<std::string, ros::Publisher> rosPublisher;
	std::map<std::string, ros::Subscriber> rosSubscriber;
};

////////////////////////////////////////////////////////////////////////////////

