/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TestXMLMacroDefine.C
 *    Tests <macro:Define_...> usage.
 *
 * @author Adrian Kriegel
 * @date   Mon Feb 10 2025
 */

#include <boost/test/unit_test.hpp>

#include <xml/XMLDomPreprocessor.h>
#include <transform/Pose.h>

#include <xml/macros/UserDefinedMacro.h>
#include <xml/macros/XMLMacroProcessor.h>

#include <xml/macros/TestUtils.h>

using namespace mira;
using namespace mira::xmlmacros;

struct MacroTestStruct
{
	int number;

	Pose2 pose1;

	Pose2 pose2;

	struct Complex
	{
		std::string foo;

		template<typename Reflector>
		void reflect(Reflector& r)
		{
			r.property("Foo", foo, "Value for Foo", "some default value");
		}
	};

	Complex complex;
	Complex copyOfComplex;

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Number", number, "Number");
		r.property("Complex", complex, "Complex member.");
		r.property("CopyOfComplex", copyOfComplex, "Copy of complex.");
		r.property("Pose1", pose1, "Pose1");
		r.property("Pose2", pose2, "Pose2");
	}
};


BOOST_AUTO_TEST_CASE(TestDefineMacro)
{
	const std::string definition = R"(
		<macro:Define_MakeTestStruct>
			<Parameters>
				<ParameterPose1 />

				<ParameterNumber>42</ParameterNumber>

				<ParameterPose2>
					<X>1</X>
					<Y>2</Y>
					<Phi>3</Phi>
				</ParameterPose2>

				<ParameterComplex Foo="Bar" />
			</Parameters>

			<!-- Create a MacroTestStruct from the parameters. -->
			<Body>
				<Pose2><macro:ParameterPose2 /></Pose2>
				<Pose1><macro:ParameterPose1 /></Pose1>
				<Number><macro:ParameterNumber /></Number>
				<Complex><macro:ParameterComplex /></Complex>
				<CopyOfComplex><macro:ParameterComplex /></CopyOfComplex>
			</Body>
		</macro:Define_MakeTestStruct>
	)";

	XMLDom dom;
	dom.loadFromString(wrapInRoot(definition + "<macro:MakeTestStruct />"));

	// Macro expansion itself should fail even before the TestStruct is deserialized if parameters are missing.
	BOOST_CHECK_THROW(XMLDomPreprocessor().preprocessXML(dom), XMissingParameter);

	// Now with all required parameters.
	dom.loadFromString(wrapInRoot(definition + R"(
		<macro:MakeTestStruct>
			<ParameterPose1>
				<X>42</X>
				<Y>0</Y>
				<Phi>180</Phi>
			</ParameterPose1>
		</macro:MakeTestStruct>
	)"));

	XMLDomPreprocessor().preprocessXML(dom);

	MacroTestStruct testStruct;
	XMLDeserializer deserializer(dom);
	deserializer.deserializeFromNode("root", testStruct);

	BOOST_CHECK_EQUAL(testStruct.number, 42);
	BOOST_CHECK_CLOSE(testStruct.pose1.x(), 42, 1e-3);
	BOOST_CHECK_CLOSE(testStruct.pose1.y(), 0, 1e-3);
	BOOST_CHECK_CLOSE(testStruct.pose1.phi(), M_PI, 1e-3);

	BOOST_CHECK_CLOSE(testStruct.pose2.x(), 1, 1e-3);
	BOOST_CHECK_CLOSE(testStruct.pose2.y(), 2, 1e-3);
	BOOST_CHECK_CLOSE(testStruct.pose2.phi() / M_PI * 180, 3, 1e-3);
}


/**
 * Tests that, while parameters passed are evaluated lazily in the body, they still refer
 * to the original scope (calling scope).
 * 
 * In this test, we define a macro that has the same name as a parameter in another macro.
 */
BOOST_AUTO_TEST_CASE(TestPassedParameterScope) 
{
	testEqual(
		R""(
		<macro:Define_Foo>
			<Parameters>
				<MyParam>abc</MyParam>
			</Parameters>
			<Body>
				<macro:MyParam />
			</Body>
		</macro:Define_Foo>

		<macro:Define_MyParam>
			<Body>123</Body>
		</macro:Define_MyParam>

		<macro:Foo><MyParam><macro:MyParam /></MyParam></macro:Foo>
		)"",
		R""(123)""
	);
}

/**
 * Tests invocation of macros with partial selection.
 */
BOOST_AUTO_TEST_CASE(TestPartialSelection) 
{
	testEqual(
		R""(
		<macro:Define_Foo>
			<Parameters>
				<MyParam />
			</Parameters>
			<Body>
				<MyNode>
					<macro:MyParam />
				</MyNode>
			</Body>
		</macro:Define_Foo>
		
		<macro:Foo MyParam="123" />
		<macro:Foo.MyNode MyParam="123" />
		<macro:Foo.MyNode><MyParam><MyNode2>123</MyNode2></MyParam></macro:Foo.MyNode>
		<macro:Foo.MyNode.MyNode2><MyParam><MyNode2>123</MyNode2></MyParam></macro:Foo.MyNode.MyNode2>
		)"",
		R""(<MyNode>123</MyNode>123<MyNode2>123</MyNode2>123)""
	);

	testEqual(
		R""(
		<macro:Define_Foo>
			<Parameters>
				<MyParam />
			</Parameters>
			<Body>
				<macro:MyParam.X />
			</Body>
		</macro:Define_Foo>
		
		<macro:Foo><MyParam><X>123</X></MyParam></macro:Foo>
		)"",
		R""(123)""
	);

	testEqual(
		R""(
		<macro:Define_Foo>
			<Parameters>
				<MyParam />
			</Parameters>
			<Body>
				<Foo><macro:MyParam.X /></Foo>
				<Bar>BAR</Bar>
			</Body>
		</macro:Define_Foo>
		
		<macro:Foo.Bar><MyParam><X>123</X></MyParam></macro:Foo.Bar>
		<macro:Foo.Foo><MyParam><X>123</X></MyParam></macro:Foo.Foo>
		)"",
		R""(BAR123)""
	);
}

/**
 * Tests that parameter defaults are evaluated in the same scope as the body.
 */
BOOST_AUTO_TEST_CASE(TestParameterDefaultScope) 
{
	testEqual(
		R""(
		<macro:Define_Foo>
			<Parameters>
				<Name>Bar</Name>
				<ID><macro:Name /></ID>
			</Parameters>
			<Body>
				<Name><macro:Name /></Name>
				<ID><macro:ID /></ID>
			</Body>
		</macro:Define_Foo>

		<macro:Foo />
		<macro:Foo Name="Baz" />
		<macro:Foo Name="Qux" ID="1234" />
		)"",
		R""(
			<Name>Bar</Name>
			<ID>Bar</ID>

			<Name>Baz</Name>
			<ID>Baz</ID>

			<Name>Qux</Name>
			<ID>1234</ID>
		)""
	);
}