/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationEigenTest.C
 *    Test for serialization framework.
 *
 * @author Ronny Stricker
 */

#include <boost/test/unit_test.hpp>

#include <math/IsApprox.h>
#include "CommonTest.h"

#include <error/LogTxtStreamSink.h>
#include <serialization/adapters/Eigen/Eigen>
#include <serialization/adapters/Eigen/SparseEigen>

using namespace std;
using namespace mira;

/// Use BOOST_CHECK(isApprox()) with precision parameter for float-type matrices
template <typename T, bool isInt = boost::is_integral<typename T::Scalar>::value>
struct Checker
{
	static void check(const T& value1, const T& value2, float precision)
	{
		BOOST_CHECK( isApprox<typename T::Scalar>(value1.squaredNorm(), value2.squaredNorm(), precision) );
		BOOST_CHECK( isApprox<typename T::Scalar>((value1 - value2).sum(), 0, precision) );
	}
};

/// Use BOOST_CHECK_EQUAL (ignore precision) for int-type matrices
template <typename T>
struct Checker<T, true>
{
	static void check(const T& value1, const T& value2, float precision)
	{
		BOOST_CHECK_EQUAL( value1.squaredNorm(), value2.squaredNorm() );
		BOOST_CHECK_EQUAL( (value1 - value2).sum(), 0 );
	}
};


template <typename T>
void serializeAll(T& value, const char* baseName, int num, float precision = 0.f)
{
	string xmlfilename = (boost::format("%s%d.xml") % baseName % num).str();

	{
		XMLDom xml;
		XMLSerializer s(xml);
		s.serialize("value", value);
		xml.saveToFile(xmlfilename);
	}

	{
		// deserialize test class
		T value2;
		XMLDom xml2;
		xml2.loadFromFile(xmlfilename);
		XMLDeserializer s2(xml2.croot());
		s2.deserialize("value", value2);

		Checker<T>::check(value, value2, precision);
	}

	{
		JSONSerializer s;
		json::Value v = s.serialize(value);

		// deserialize test class
		T value2; // with 0

		JSONDeserializer s2(v);
		s2.deserialize(value2);

		Checker<T>::check(value, value2, precision);
	}

	{
		BinaryBufferOstream::buffer_type buffer;
		BinaryBufferSerializer s(&buffer);
		s.serialize(value);

		// deserialize test class
		T value2; // with 0

		BinaryBufferDeserializer s2(&buffer);
		s2.deserialize(value2);

		Checker<T>::check(value, value2, precision);
	}
}

//////////////////////////////////////////////////////////////////////////////

typedef Eigen::SparseMatrix<int> SMat;
typedef Eigen::Matrix<int,Eigen::Dynamic,Eigen::Dynamic> DMat;
typedef Eigen::Matrix<double,Eigen::Dynamic,Eigen::Dynamic> DDMat;

BOOST_AUTO_TEST_CASE( SparseTest )
{
	SMat V(5,5);

	V.insert(1,1) = 1;
	V.insert(2,3) = 2;
	V.insert(4,1) = 3;

	serializeAll(V, "Sparse", 1);
}

BOOST_AUTO_TEST_CASE( DenseTest )
{
	DMat V(5,5);
	V.setZero();

	V(1,1) = 1;
	V(2,3) = 2;
	V(4,1) = 3;

	serializeAll(V, "Dense", 1);
}

BOOST_AUTO_TEST_CASE( PrecisionTest )
{
	DDMat V(5,5);
	V.setZero();

	V(1,1) = 1.0/3.0;
	V(2,3) = 1.0/7.0;
	V(4,1) = 5.0/9.0;

	serializeAll(V, "Precision", 1, 1e-15);
}

//////////////////////////////////////////////////////////////////////////////
