/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationCvMatTest.C
 *    Test for serialization framework.
 *
 * @author Christof Schröter
 * @date   2020/09/16
 */

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/opencv2/core/core.hpp>
#include <serialization/XMLSerializer.h>
#include <serialization/BinarySerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/Print.h>

using namespace mira;

//////////////////////////////////////////////////////////////////////////////

void checkEqual(const cv::Mat& mat1, const cv::Mat& mat2) {
	BOOST_CHECK_EQUAL(mat1.size(), mat2.size());

	for (int r = 0; r < mat1.rows; ++r) {
		for (int c = 0; c < mat1.cols; ++c) {
			for (uint32 b = 0; b < mat1.channels()*mat1.elemSize1(); ++b) {
				// byte by byte, so we do not need to determine the type here
				BOOST_CHECK_EQUAL(*(mat1.ptr<uint8>(r,c)+b), *(mat2.ptr<uint8>(r,c)+b));
				if (*(mat1.ptr<uint8>(r,c)+b) != *(mat2.ptr<uint8>(r,c)+b)) {
					std::cout << "==========================================" << std::endl;
					std::cout << print(mat1) << std::endl;
					std::cout << "vs." << std::endl;
					std::cout << print(mat2) << std::endl;
					std::cout << "==========================================" << std::endl;
					return;
				}
			}
		}
	}
}

void testCvMat(const cv::Mat& mat)
{
	XMLDom xml;
	XMLSerializer s(xml);

	s.serialize("Mat", mat, "");
//	xml.saveToFile("out.xml");

	XMLDeserializer ds(xml);
	cv::Mat mat2;
	ds.deserialize("Mat", mat2);

//	std::cout << "xml" << std::endl;
	checkEqual(mat, mat2);

	Buffer<uint8> buffer;
	BinaryBufferSerializer bs(&buffer);
	bs.serialize(mat,false);

	cv::Mat mat3;
	BinaryBufferDeserializer bd(&buffer);
	bd.deserialize(mat3,false);

//	std::cout << "binary" << std::endl;
	checkEqual(mat, mat3);

	JSONSerializer js;
 	json::Value v = js.serialize(mat);

	JSONDeserializer jd(v);
	cv::Mat mat4;
 	jd.deserialize(mat4);

// 	std::cout << "json" << std::endl;
 	checkEqual(mat, mat4);
}

BOOST_AUTO_TEST_CASE( JSONTest )
{
	cv::Mat mi(3,3, CV_8UC1);
	std::cout << print(mi) << std::endl;

	cv::Mat mf(3,3, CV_32FC3);
//	std::cout << print(mf) << std::endl;

	testCvMat(cv::Mat( 5,  5, CV_8UC1 ));
	testCvMat(cv::Mat( 4,  6, CV_16SC3));

	// without initialization, float matrices can have NaN values that are not reflected correctly

	cv::Mat m32(10, 10, CV_32FC1);
	m32 = cv::Scalar(-3e5, 0.f, -555.f);
	testCvMat(m32);

	cv::Mat m64(10, 10, CV_64FC3);
	m64 = cv::Scalar(1.0, 1.7, -4.2);
	testCvMat(m64);
}

//////////////////////////////////////////////////////////////////////////////
