/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationContainerExportTest.C
 *    Test for serialization framework.
 *
 * @author Christof Schröter
 */

#include <boost/test/unit_test.hpp>

#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/set>
#include <serialization/adapters/std/map>
#include <serialization/adapters/std/unordered_map>

#include <serialization/XMLSerializer.h>
#include <serialization/JSONSerializer.h>
#include <serialization/PropertySerializer.h>

using namespace std;
using namespace mira;

//////////////////////////////////////////////////////////////////////////////

template <typename Container>
void CHECK_EQUAL(const Container& c1, const Container& c2)
{
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	typename Container::const_iterator i1 = c1.begin(), i2 = c2.begin();

	for (; i1 != c1.end(); ++i1, ++i2)
		BOOST_CHECK_EQUAL(*i1, *i2);
}

template <typename T>
void CHECK_EQUAL(const map<T,T>& c1, const map<T,T>& c2)
{
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	typename map<T,T>::const_iterator i1 = c1.begin(), i2 = c2.begin();

	for (; i1 != c1.end(); ++i1, ++i2) {
		BOOST_CHECK_EQUAL(i1->first, i2->first);
		BOOST_CHECK_EQUAL(i1->second, i2->second);
	}
}

template <typename T>
void CHECK_EQUAL(const unordered_map<T,T>& c1, const unordered_map<T,T>& c2)
{
	BOOST_CHECK_EQUAL(c1.size(), c2.size());
	// just check the same keys exist and have the same values, disregarding order
	for (const auto& [key, value] : c1) {
		BOOST_CHECK_EQUAL(value, c2.at(key));
	}
}

///////////////////////////////////////////////////////////////////////////////

void insert(vector<int>& container, int value)
{
	container.push_back(value);
}

void insert(vector<string>& container, int& value)
{
	container.push_back(toString(value));
}

void insert(set<int>& container, int value)
{
	container.insert(value);
}

void insert(set<string>& container, int value)
{
	container.insert(toString(value));
}

template <template<typename U, typename V> class MapType>
void insert(MapType<int,int>& container, int value)
{
	container[value] = 2 * value;
}

template <template<typename U, typename V> class MapType>
void insert(MapType<string,string>& container, int value)
{
	container[toString(value)] = "-"+toString(value)+"-";
}

///////////////////////////////////////////////////////////////////////////////

template<typename T>
void read(vector<T>& container, const json::Value& v)
{
	int length = v.get_array().size();
	for (int n = 0; n < length; ++n)
		container.push_back(v.get_array()[n].get_value<typename json::TypeTrait<T>::type >());
}

template<typename T>
void read(set<T>& container, const json::Value& v)
{
	int length = v.get_array().size();
	for (int n = 0; n < length; ++n)
		container.insert(v.get_array()[n].get_value<typename json::TypeTrait<T>::type >());
}

template<template<typename U, typename V> class MapType, typename T>
void readMapStandard(MapType<T,T>& container, const json::Value& v)
{
	int length = v.get_array().size();
	for (int n = 0; n < length; ++n) {
		json::Value vp = v.get_array()[n];
		T key = vp.get_obj()["First"].get_value<typename json::TypeTrait<T>::type >();
		T val = vp.get_obj()["Second"].get_value<typename json::TypeTrait<T>::type >();
		container[key] = val;
	}
}

template<template<typename U, typename V> class MapType, typename T>
void readMapFromObject(MapType<string,T>& container, const json::Value& v)
{
	json::Object o = v.get_obj();
	foreach (auto e, o) {
		string key = e.first;
		T val = e.second.get_value<typename json::TypeTrait<T>::type >();
		container[key] = val;
	}
}

template<template<typename V, typename W> class MapType, typename T, typename U>
void read(MapType<T,U>& container, const json::Value& v)
{
	readMapStandard(container, v);
}

template<template<typename U, typename V> class MapType, typename T>
void read(MapType<string,T>& container, const json::Value& v, bool fromObject)
{
	if (fromObject)
		readMapFromObject(container, v);
	else
		readMapStandard(container, v);
}

template<typename Container>
void testJSON(int length, bool pretty=false)
{
	Container c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	JSONSerializer s;
	stringstream ss;

	json::write(s.serialize(c1), ss, pretty);
	cout << ss.str() << endl;

	json::Value v;

	json::read(ss.str(), v);
	read(c2, v);
	CHECK_EQUAL(c1, c2);
}

template<template<typename U, typename V> class MapType, typename T>
void testStringMapJSON(int length, bool pretty=false, bool asObject=false)
{
	MapType<string, T> c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	JSONSerializer s(false, asObject ? JSONSerializer::STRING_MAP_AS_OBJECT : JSONSerializer::STANDARD);
	stringstream ss;

	json::write(s.serialize(c1), ss, pretty);
	cout << ss.str() << endl;

	json::Value v;

	json::read(ss.str(), v);
	read(c2, v, asObject);
	CHECK_EQUAL(c1, c2);
}

BOOST_AUTO_TEST_CASE( TestSequenceToJSON )
{
	testJSON<vector<int>>(5);
	testJSON<vector<int>>(5, true);
	testJSON<vector<string>>(5);
	testJSON<vector<string>>(5, true);
}

BOOST_AUTO_TEST_CASE( TestSetToJSON )
{
	testJSON<set<int>>(5);
	testJSON<set<int>>(5, true);
	testJSON<set<string>>(5);
	testJSON<set<string>>(5, true);
}

BOOST_AUTO_TEST_CASE( TestMapToJSON )
{
	testJSON<map<int,int>>(5);
	testJSON<map<int,int>>(5, /*pretty=*/ true);

	testStringMapJSON<map, string>(5, /*pretty=*/ false, /*asObject=*/ false);
	testStringMapJSON<map, string>(5,             false,               true);
	testStringMapJSON<map, string>(5,             true,                false);
	testStringMapJSON<map, string>(5,             true,                true);

	testStringMapJSON<unordered_map, string>(5, /*pretty=*/ false, /*asObject=*/ false);
	testStringMapJSON<unordered_map, string>(5,             false,               true);
	testStringMapJSON<unordered_map, string>(5,             true,                false);
	testStringMapJSON<unordered_map, string>(5,             true,                true);
}

///////////////////////////////////////////////////////////////////////////////

template<typename T>
void read(vector<T>& container, const XMLDom& xml)
{
	XMLDom::const_sibling_iterator xmlroot = xml.croot();
	XMLDom::const_sibling_iterator node;
	XMLDom::const_content_iterator content;

	node = xmlroot.find("test");
	BOOST_CHECK(node != xmlroot.cend()); // tag 'test' must exist
	node = node.cbegin();
	while (node != xmlroot.cend()) {     // 'test' can contain 0 or more tags 'item'
		BOOST_CHECK_EQUAL(*node, "item"); // and nothing else
		content = node.content_cbegin();
		BOOST_CHECK(content != node.content_cend());
		container.push_back(boost::lexical_cast<T>(*content));
		++node;
	}
}

template<typename T>
void read(set<T>& container, const XMLDom& xml)
{
	XMLDom::const_sibling_iterator xmlroot = xml.croot();
	XMLDom::const_sibling_iterator node;
	XMLDom::const_content_iterator content;

	node = xmlroot.find("test");
	BOOST_CHECK(node != xmlroot.cend());
	node = node.cbegin();
	while (node != xmlroot.cend()) {
		BOOST_CHECK_EQUAL(*node, "item");
		content = node.content_cbegin();
		BOOST_CHECK(content != node.content_cend());
		container.insert(boost::lexical_cast<T>(*content));
		++node;
	}
}

template<typename T>
void read(map<T,T>& container, const XMLDom& xml)
{
	XMLDom::const_sibling_iterator xmlroot = xml.croot();
	XMLDom::const_sibling_iterator node;
	XMLDom::const_sibling_iterator pairnode;
	XMLDom::const_content_iterator content;

	node = xmlroot.find("test");
	BOOST_CHECK(node != xmlroot.cend());
	node = node.cbegin();
	while (node != xmlroot.cend()) {
		BOOST_CHECK_EQUAL(*node, "key");  // must have key/item pairs
		content = node.content_cbegin();
		BOOST_CHECK(content != node.content_cend());
		T key = boost::lexical_cast<T>(*content);
		++node;
		BOOST_CHECK(node != xmlroot.cend());
		BOOST_CHECK_EQUAL(*node, "item");
		content = node.content_cbegin();
		BOOST_CHECK(content != node.content_cend());
		T value = boost::lexical_cast<T>(*content);
		container[key] = value;
		++node;		
	}
}

template<typename Container>
void testXML(int length)
{
	Container c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	XMLDom xml;
	XMLSerializer s(xml);
	
	s.serialize("test", c1);
	string sxml = xml.saveToString();
	cout << sxml << endl;

	XMLDom xml2;

	xml2.loadFromString(sxml);
	read(c2, xml2);
	CHECK_EQUAL(c1, c2);	
}

BOOST_AUTO_TEST_CASE( TestSequenceToXML )
{
	testXML<vector<int>>(5);
	testXML<vector<string>>(5);
}

BOOST_AUTO_TEST_CASE( TestSetToXML )
{
	testXML<set<int>>(5);
	testXML<set<string>>(5);
}

BOOST_AUTO_TEST_CASE( TestMapToXML )
{
	testXML<map<int,int>>(5);
	testXML<map<string, string>>(5);
}

///////////////////////////////////////////////////////////////////////////////

template<typename T>
void read(vector<T>& container, const PropertyNode::NodeList& nodes)
{
	foreach (const PropertyNode* p, nodes) {
		cout << p->getAsString() << endl;
		container.push_back(boost::lexical_cast<T>(p->getAsString()));
	}
}

template<typename T>
void read(set<T>& container, const PropertyNode::NodeList& nodes)
{
	foreach (const PropertyNode* p, nodes) {
		cout << p->getAsString() << endl;
		container.insert(boost::lexical_cast<T>(p->getAsString()));
	}
}

template<typename T>
void read(map<T,T>& container, const PropertyNode::NodeList& nodes)
{
	int length = nodes.size();
	BOOST_CHECK_EQUAL(length % 2, 0);
	for (int n = 0; n < length / 2; ++n) {
		T key = boost::lexical_cast<T>(nodes[2*n]->getAsString());
		T val = boost::lexical_cast<T>(nodes[2*n+1]->getAsString());
		cout << key << ": " << val << endl;
		container[key] = val;
	}
}

template<typename Container>
void testProperties(int length)
{
	Container c1, c2;
	for (int n = 0; n < length; ++n)
		insert(c1, n);

	PropertySerializer s;
	
	PropertyNode* node = s.reflectProperties("test", c1);
	read(c2, node->children());
	CHECK_EQUAL(c1, c2);	
}

BOOST_AUTO_TEST_CASE( TestVectorToProperties )
{
	testProperties<vector<int>>(5);
	testProperties<vector<string>>(5);
}

BOOST_AUTO_TEST_CASE( TestSetToProperties )
{
	testProperties<set<int>>(5);
	testProperties<set<string>>(5);
}

BOOST_AUTO_TEST_CASE( TestMapToProperties )
{
	testProperties<map<int,int>>(5);
	testProperties<map<string, string>>(5);
}

///////////////////////////////////////////////////////////////////////////////
