/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RSASignatureTest.C
 *    Test case for RSASignature.
 *
 * @author Christian Martin
 * @date   2011/01/14
 */

#include <boost/test/unit_test.hpp>

#include <security/RSASignature.h>

#include <utils/ToString.h>

using namespace mira;
using namespace std;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSASignatureTest_ShortMsg )
{
	RSASignature::DigestType digest[4] = {
		RSASignature::DIGEST_MD5,
		RSASignature::DIGEST_SHA1,
		RSASignature::DIGEST_SHA256,
		RSASignature::DIGEST_SHA512
	};

	foreach(auto digestType, digest) {
		RSAKey publicKey, privateKey;
		RSAKey::generateKey(2048, publicKey, privateKey);

		std::string msg("This is a test message.");
		std::string signatureStr;

		{ // sign message with private key
			RSASignature signature =
				RSASignature::signMessage(privateKey, digestType,
				                          msg.c_str(), msg.size());
			signatureStr = toString(signature);
		}

		{ // verify message by using the public key (normal use-case)
			RSASignature signature = fromString<RSASignature>(signatureStr);
			BOOST_CHECK(RSASignature::verifyMessage(publicKey, digestType,
			                                        msg.c_str(), msg.size(),
			                                        signature));
		}

		{ // verify message by using the public key constructed from N and E
			RSAKey keyNE(publicKey.getNStr(), publicKey.getEStr(), "");
			RSASignature signature = fromString<RSASignature>(signatureStr);
			BOOST_CHECK(RSASignature::verifyMessage(keyNE, digestType,
			                                        msg.c_str(), msg.size(),
			                                        signature));
		}

		{ // verify message by using the private key (also must work!)
			RSASignature signature = fromString<RSASignature>(signatureStr);
			BOOST_CHECK(RSASignature::verifyMessage(privateKey, digestType,
			                                        msg.c_str(), msg.size(),
			                                        signature));
		}
}
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( RSASignatureTest_LongMsg )
{
	RSASignature::DigestType digest[4] = {
		RSASignature::DIGEST_MD5,
		RSASignature::DIGEST_SHA1,
		RSASignature::DIGEST_SHA256,
		RSASignature::DIGEST_SHA512
	};

	foreach(auto digestType, digest) {
		RSAKey publicKey, privateKey;
		RSAKey::generateKey(2048, publicKey, privateKey);

		string msg;
		for(int i = 0; i < 2000; i++)
			msg += "0123456789abcdefghijklmnopqrstuvwxyz";

		std::string signatureStr;

		{ // sign message with private key
			RSASignature signature =
				RSASignature::signMessage(privateKey, digestType,
				                          msg.c_str(), msg.size());
			signatureStr = toString(signature);
		}

		{ // verify message by using the public key
			RSASignature signature = fromString<RSASignature>(signatureStr);
			BOOST_CHECK(RSASignature::verifyMessage(publicKey, digestType,
			                                        msg.c_str(), msg.size(),
			                                        signature));
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
