/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RasterTransformTest.C
 *    A test case for the RasterTransformation.
 *
 * @author Christof Schröter
 * @date   2017/02/14
 */

#include <iostream>
#include <boost/test/unit_test.hpp>
#include <geometry/RasterTransformation.h>
#include <math/Random.h>
#include <image/Img.h>

using namespace std;
using namespace mira;

#define RANDOMTEST_CYCLES 1000

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( DefinedRotationTest )
{
	Img8U3 src(1024, 768);

	// random init
	for (int y = 0; y < src.height(); ++y)
	{
		unsigned char* data = src.data(y);
		for (int x = 0; x < src.width(); ++x)
			data[x] = MIRA_RANDOM.uniform(0,256);
	}

	// tgt size is rotated
	Img8U3 tgt(src.height(), src.width());

	// just rotate by 90° (src -> tgt)
	RasterTransformation rt_fw(Rect2i(0,0,src.width()+1,src.height()+1), // using size+1 here to also get the last line
	                           Rect2i(0,0,tgt.width()+1,tgt.height()+1), // -> copy the entire image (not guaranteed otherwise)
	                           Point2d(src.width()/2,src.height()/2.f),
	                           Point2d(tgt.width()/2.f,tgt.height()/2.f),
	                           1.0,
	                           boost::math::constants::half_pi<double>(),
	                           true);

	{
		int minSrcX = 10000, maxSrcX = -1, minSrcY = 10000, maxSrcY = -1;
		int minTgtX = 10000, maxTgtX = -1, minTgtY = 10000, maxTgtY = -1;
		for (RasterTransformation::iterator it = rt_fw.begin(); it.isValid(); ++it)
		{
			minSrcX = std::min(minSrcX, it.srcX());
			maxSrcX = std::max(maxSrcX, it.srcX());
			minSrcY = std::min(minSrcY, it.srcY());
			maxSrcY = std::max(maxSrcY, it.srcY());
			minTgtX = std::min(minTgtX, it.tgtX());
			maxTgtX = std::max(maxTgtX, it.tgtX());
			minTgtY = std::min(minTgtY, it.tgtY());
			maxTgtY = std::max(maxTgtY, it.tgtY());

			tgt(it.tgtX(), it.tgtY()) = src(it.srcX(), it.srcY());
		}

		BOOST_CHECK_EQUAL(minSrcX, 0);
		BOOST_CHECK_EQUAL(maxSrcX, src.width()-1);
		BOOST_CHECK_EQUAL(minSrcY, 0);
		BOOST_CHECK_EQUAL(maxSrcY, src.height()-1);

		BOOST_CHECK_EQUAL(minTgtX, 0);
		BOOST_CHECK_EQUAL(maxTgtX, tgt.width()-1);
		BOOST_CHECK_EQUAL(minTgtY, 0);
		BOOST_CHECK_EQUAL(maxTgtY, tgt.height()-1);
	}

	// same size as original image
	Img8U3 tgt2(src.size());

	// rotate back (tgt -> tgt2)
	RasterTransformation rt_bw(Rect2i(0,0,tgt.width()+1,tgt.height()+1),
	                           Rect2i(0,0,tgt2.width()+1,tgt2.height()+1),
	                           Point2d(tgt.width()/2,tgt.height()/2),
	                           Point2d(tgt2.width()/2,tgt2.height()/2),
	                           1.0,
	                           -boost::math::constants::half_pi<double>(),
	                           true);

	{
		int minSrcX = 10000, maxSrcX = -1, minSrcY = 10000, maxSrcY = -1;
		int minTgtX = 10000, maxTgtX = -1, minTgtY = 10000, maxTgtY = -1;
		for (RasterTransformation::iterator it = rt_bw.begin(); it.isValid(); ++it)
		{
			minSrcX = std::min(minSrcX, it.srcX());
			maxSrcX = std::max(maxSrcX, it.srcX());
			minSrcY = std::min(minSrcY, it.srcY());
			maxSrcY = std::max(maxSrcY, it.srcY());
			minTgtX = std::min(minTgtX, it.tgtX());
			maxTgtX = std::max(maxTgtX, it.tgtX());
			minTgtY = std::min(minTgtY, it.tgtY());
			maxTgtY = std::max(maxTgtY, it.tgtY());

			tgt2(it.tgtX(), it.tgtY()) = tgt(it.srcX(), it.srcY());
		}
		BOOST_CHECK_EQUAL(minSrcX, 0);
		BOOST_CHECK_EQUAL(maxSrcX, tgt.width()-1);
		BOOST_CHECK_EQUAL(minSrcY, 0);
		BOOST_CHECK_EQUAL(maxSrcY, tgt.height()-1);

		BOOST_CHECK_EQUAL(minTgtX, 0);
		BOOST_CHECK_EQUAL(maxTgtX, tgt2.width()-1);
		BOOST_CHECK_EQUAL(minTgtY, 0);
		BOOST_CHECK_EQUAL(maxTgtY, tgt2.height()-1);
	}

	BOOST_CHECK(src == tgt2);

}

void info(const Rect2i& src, const Rect2i& tgt, const Point2d& srcRef, const Point2d& tgtRef,
          double scale, double rot)
{
	std::cout << "src: " << src.minCorner.x() << "," << src.minCorner.y() << " - "
	                     << src.maxCorner.x() << "," << src.maxCorner.y() << std::endl;
	std::cout << "tgt: " << tgt.minCorner.x() << "," << tgt.minCorner.y() << " - "
	                     << tgt.maxCorner.x() << "," << tgt.maxCorner.y() << std::endl;
	std::cout << "srcRef: " << srcRef.x() << "," << srcRef.y() << std::endl;
	std::cout << "tgtRef: " << tgtRef.x() << "," << tgtRef.y() << std::endl;
	std::cout << "scale: " << scale << std::endl;
	std::cout << "rot: " << rot << " (" << rad2deg(rot) << "deg)"  << std::endl;
}

void testRT(const Rect2i& src, const Rect2i& tgt,
            double scale, double rot)
{
	// src ref point up to 10% of size outside src
	Point2d srcRef(MIRA_RANDOM.uniform(src.x0() - src.width() * 0.1,
	                                   src.x0() + src.width() * 1.1),
	               MIRA_RANDOM.uniform(src.y0() - src.height() * 0.1,
	                                   src.y0() + src.height() * 1.1));

	// tgt ref point up to 10% of size outside tgt
	Point2d tgtRef(MIRA_RANDOM.uniform(tgt.x0() - tgt.width() * 0.1,
	                                   tgt.x0() + tgt.width() * 1.1),
	               MIRA_RANDOM.uniform(tgt.y0() - tgt.height() * 0.1,
	                                   tgt.y0() + tgt.height() * 1.1));

	RasterTransformation rt(src, tgt, srcRef, tgtRef, scale, rot, true);

	for (RasterTransformation::iterator it = rt.begin(); it.isValid(); ++it)
	{
		if ((it.srcX() < src.x0())  ||
		    (it.srcX() >= src.x1()) ||
		    (it.srcY() < src.y0())  ||
		    (it.srcY() >= src.y1()) ||
		    (it.tgtX() < tgt.x0())  ||
		    (it.tgtX() >= tgt.x1()) ||
		    (it.tgtY() < tgt.y0())  ||
		    (it.tgtY() >= tgt.y1()))
		{
			info(src, tgt, srcRef, tgtRef, scale, rot);
		}

		BOOST_CHECK_GE(it.srcX(), src.x0());
		BOOST_CHECK_LT(it.srcX(), src.x1());
		BOOST_CHECK_GE(it.srcY(), src.y0());
		BOOST_CHECK_LT(it.srcY(), src.y1());

		BOOST_CHECK_GE(it.tgtX(), tgt.x0());
		BOOST_CHECK_LT(it.tgtX(), tgt.x1());
		BOOST_CHECK_GE(it.tgtY(), tgt.y0());
		BOOST_CHECK_LT(it.tgtY(), tgt.y1());
	}
}

void testRTVariate(const Point2i& srcMinCorner, const Point2i& tgtMinCorner,
                   double scale, double rot)
{
	// for each src/target area min corner, test all combinations of
	// width and height 0,1,2,random(3, 100)
	std::vector<Rect2i> srcVariations, tgtVariations;

	for (int w = 0; w <= 2; ++w) {
		for (int h = 0; h <= 2; ++h) {
			srcVariations.emplace_back(srcMinCorner, w, h);
			tgtVariations.emplace_back(tgtMinCorner, w, h);
		}
		srcVariations.emplace_back(srcMinCorner, w, MIRA_RANDOM.uniform(3,100));
		srcVariations.emplace_back(srcMinCorner, MIRA_RANDOM.uniform(3,100), w);
		tgtVariations.emplace_back(tgtMinCorner, w, MIRA_RANDOM.uniform(3,100));
		tgtVariations.emplace_back(tgtMinCorner, MIRA_RANDOM.uniform(3,100), w);
	}

	srcVariations.emplace_back(srcMinCorner,
	                           MIRA_RANDOM.uniform(3,100),
	                           MIRA_RANDOM.uniform(3,100));
	tgtVariations.emplace_back(tgtMinCorner,
	                           MIRA_RANDOM.uniform(3,100),
	                           MIRA_RANDOM.uniform(3,100));

	foreach(const auto& s, srcVariations) {
		foreach(const auto& t, tgtVariations) {
			testRT(s, t, scale, rot);
		}
	}
}

BOOST_AUTO_TEST_CASE( LimitsTest )
{
	for (int n = 1; n < RANDOMTEST_CYCLES; ++n) {

		Point2i src(MIRA_RANDOM.uniform(0,100), MIRA_RANDOM.uniform(0,100));
		Point2i tgt(MIRA_RANDOM.uniform(0,100), MIRA_RANDOM.uniform(0,100));

		double scale = MIRA_RANDOM.uniform(0.5, 2.0);
		double rot = MIRA_RANDOM.uniform(-boost::math::constants::pi<double>(),
		                                  boost::math::constants::pi<double>());

		testRTVariate(src, tgt, scale, rot);
	}
}

RasterTransformation makeIdRT(bool select)
{
	Rect2i src(0, 0, 100, 100);
	Rect2i tgt(0, 0, 100, 100);

	Point2d srcRef(50, 50);
	Point2d tgtRef(50, 50);

	double scale = 1;
	double rot = 0;

	RasterTransformation rt(src, tgt, srcRef, tgtRef, scale, rot, false);
	std::cout << &rt << std::endl;
	BOOST_CHECK_EQUAL(rt.begin().mT, &rt);

	RasterTransformation rt2(src, tgt, srcRef, tgtRef, scale, rot, false);
	std::cout << &rt2 << std::endl;
	BOOST_CHECK_EQUAL(rt2.begin().mT, &rt2);

	// force copy! (disable copy elision)
	if (select)
		return rt;
	else
		return rt2;
}

BOOST_AUTO_TEST_CASE( CopyTest )
{
	RasterTransformation rt = makeIdRT(false);
	std::cout << &rt << std::endl;
	BOOST_CHECK_EQUAL(rt.begin().mT, &rt);

	int minSrcX = 100, maxSrcX = -1, minSrcY = 100, maxSrcY = -1;
	int minTgtX = 100, maxTgtX = -1, minTgtY = 100, maxTgtY = -1;

	for (RasterTransformation::iterator it = rt.begin(); it.isValid(); ++it)
	{
		BOOST_CHECK_EQUAL(it.srcX(), it.tgtX());
		BOOST_CHECK_EQUAL(it.srcY(), it.tgtY());

		minSrcX = std::min(minSrcX, it.srcX());
		maxSrcX = std::max(maxSrcX, it.srcX());
		minSrcY = std::min(minSrcY, it.srcY());
		maxSrcY = std::max(maxSrcY, it.srcY());
		minTgtX = std::min(minTgtX, it.tgtX());
		maxTgtX = std::max(maxTgtX, it.tgtX());
		minTgtY = std::min(minTgtY, it.tgtY());
		maxTgtY = std::max(maxTgtY, it.tgtY());
	}

	BOOST_CHECK_EQUAL(minSrcX, 0);
	BOOST_CHECK_GE(maxSrcX, 98);
	BOOST_CHECK_EQUAL(minSrcY, 0);
	BOOST_CHECK_GE(maxSrcY, 98);

	BOOST_CHECK_EQUAL(minTgtX, 0);
	BOOST_CHECK_GE(maxTgtX, 98);
	BOOST_CHECK_EQUAL(minTgtY, 0);
	BOOST_CHECK_GE(maxTgtY, 98);

}
