/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Utils.C
 *    Impl Utils.h
 *
 * @author Adrian Kriegel
 * @date   Mon Jan 20 2025
 */

#include <xml/macros/Utils.h>

#include <variant>

#include <xml/XMLDomReflect.h>

namespace mira::xmlmacros {

CopyableXMLDom::CopyableXMLDom(const CopyableXMLDom& dom)
{
	*this = dom;
}

CopyableXMLDom& CopyableXMLDom::operator=(const CopyableXMLDom& dom)
{
	root().replace(dom.root());

	return *this;
}

namespace internal {

template<typename T>
inline constexpr bool always_false_v = false;

}

XMLNode spreadNode(XMLNode location, XMLDom::const_iterator node, bool deleteLocation)
{
	// sadly, we cannot just iterate over the libxml nodes in order as the ++ operator 
	// for sibling_iterator and content_iterator will always return the same node type 
	// so we need to ensure insertion order matches the order of the nodes in the source document
	using Position = decltype(node.position());

	using NodeOrContent = std::variant<XMLDom::const_iterator, XMLDom::const_content_iterator>;

	std::vector<std::pair<Position, NodeOrContent>> nodesToInsert;

	for (auto it = node.cbegin(); it != node.cend(); ++it) {
		nodesToInsert.emplace_back(std::make_pair(it.position(), it));
	}

	for (auto it = node.content_begin(); it != node.content_end(); ++it) {
		nodesToInsert.emplace_back(std::make_pair(it.position(), it));
	}

	for (auto it = node.attribute_begin(); it != node.attribute_end(); ++it) {
		location.parent().add_attribute(*it);
	}

	std::sort(nodesToInsert.begin(), nodesToInsert.end(), [](const auto& a, const auto& b) {
		return a.first < b.first;
	});

	auto insertLocation = location;

	for (auto& [pos, it] : nodesToInsert) {
		
		std::visit([&location, pos = pos](auto& i) {

			using T = std::decay_t<decltype(i)>;

			if constexpr (std::is_same_v<T, XMLDom::const_iterator>) {
				location.insert_before(i);
			} else if constexpr (std::is_same_v<T, XMLDom::const_content_iterator>) {
				location.insert_content_before(*i);
			} else {
				static_assert(internal::always_false_v<T>, "Unhandled variant type.");
			}
		}, it);
	}

	if (deleteLocation) {
		location.remove();
	}

	return location.parent();
}

} // namespace mira::xmlmacros
