/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UserDefinedMacro.C
 *    Impl UserDefinedMacro.h
 *
 * @author Adrian Kriegel
 * @date   Sat Jan 18 2025
 */

#include <iterator>

#include <xml/XMLDomReflect.h>
#include <xml/XMLDomPreprocessor.h>

#include <xml/macros/UserDefinedMacro.h>

#include <serialization/ReflectControlFlags.h>
#include <serialization/XMLSerializer.h>

#include <xml/macros/Builtins.h>
#include <xml/macros/Utils.h>
#include <xml/macros/XMLMacroProcessor.h>

namespace mira::xmlmacros {

XMLNode UserDefinedMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	if (processor.getPreprocessor().expandMacrosOnly()) {
		node.insert_comment_before("defined at " + mDefinition.source.file + ":" + std::to_string(mDefinition.source.line));
	}

	Invocation invocation(mDefinition);

	XMLDeserializer deserializer(node);
	deserializer.deserializeFromNode((*node).c_str(), invocation);

	// Create a new scope for the invocation which is attached to the scope when the macro was defined.
	// The invoke scope carries all parameters that are passed in the invocation.
	const auto invokeScope = std::make_shared<Scope>(Scope{mScope, {}});

	const auto& params = mDefinition.parameters.parameters;

	assert(invocation.arguments.size() == params.size());

	for (uint i = 0; i < invocation.arguments.size(); ++i) {
		Identifier identifier = processor.makeIdentifier(params.at(i).name);
		
		const auto& arg = invocation.arguments.at(i);

		// Note that the argument scope is the invokeScope for any default arguments.
		// This means we may end up with a circular reference which requires manual cleanup later on.
		auto argumentScope = arg.isDefault ? invokeScope : processor.getCurrentScopePtr();

		invokeScope->define(std::move(identifier),
		                    std::make_shared<XMLValueMacro>(argumentScope, arg.value, arg.source));
	}

	// Copy and transform the body.
	CopyableXMLDom body = mDefinition.body;

	auto prevScope = processor.swapScope(invokeScope);

	if (body.root().begin() != body.root().end()) {
		processor.processInPlace(body.root());
	}

	processor.swapScope(prevScope);

	// invokeScope may contain definitions that point back at it.
	invokeScope->collectGarbage();

	return spreadNode(node, body.root());
}


void UserDefinedMacro::coerceToString(XMLMacroProcessor& processor, std::ostream& ss)
{
	XMLDom dom;
	auto node = expand(processor, dom.root().add_child("dummy"));

	for (auto it = dom.root().content_begin(); it != dom.root().content_end(); ++it) {
		ss << *it;
	}
}

CopyableXMLDom XMLValueMacro::evaluate(XMLMacroProcessor& processor) const
{
	auto prevScope = processor.swapScope(mScope);

	CopyableXMLDom copy = mValue;

	processor.processInPlace(copy.root());

	processor.swapScope(prevScope);

	return copy;
}

XMLNode XMLValueMacro::expand(XMLMacroProcessor& processor, XMLNode node)
{
	if (processor.getPreprocessor().expandMacrosOnly() && !mSource.file.empty()) {
		node.insert_comment_before("defined at " + mSource.file + ":" + std::to_string(mSource.line));
	}

	auto res = spreadNode(node, evaluate(processor).root());

	return res;
}

void XMLValueMacro::coerceToString(XMLMacroProcessor& processor, std::ostream& ss)
{
	auto xml = evaluate(processor);

	for (auto it = xml.root().content_begin(); it != xml.root().content_end(); ++it) {
		ss << *it;
	}
}

void Definition::reflect(XMLDeserializer& r)
{
	source = { r.getNode().uri(), r.getNode().line() };

	name = DefineMacro::extractName(r.getNode());

	r.property("Parameters", parameters, "Parameters required to invoke this macro.",
	           serialization::IgnoreMissing());

	r.property("Body", body, "What the macro expands to.");
}

void Parameter::reflect(XMLDeserializer& r)
{
	auto node = r.getNode();
	name = *node;

	const bool hasAttributes = node.attribute_begin() != node.attribute_end();
	const bool hasChildren = node.cbegin() != node.cend();
	const bool hasText = node.content_begin() != node.content_end();

	if (hasAttributes || hasChildren || hasText) {
		r.delegate(defaultValue.emplace());
	}
}

void Parameters::reflect(XMLDeserializer& r)
{
	auto node = r.getNode();

	parameters.reserve(std::distance(node.cbegin(), node.cend()));
	parameters.clear();

	for (auto it = node.cbegin(); it != node.cend(); ++it) {
		r.property((*it).c_str(), parameters.emplace_back(), "Macro parameter.");
	}
}

void Invocation::reflect(XMLDeserializer& r)
{
	const auto& params = definition->parameters.parameters;

	arguments.reserve(params.size());
	arguments.clear();

	auto node = r.getNode();

	for (const auto& param : params) {
		auto& arg = arguments.emplace_back();

		const auto attr = node.find_attribute(param.name);

		const bool definedAsAttribute = attr != node.attribute_end();
		const bool definedAsChild = node.find(param.name) != node.end();

		if (definedAsAttribute && definedAsChild) {
			MIRA_THROW(XInvalidConfig,
			           "Multiple definition of macro parameter (as attribute and as a child node) \""
			               << param.name << "\".");
		}

		if (!param.defaultValue && !definedAsAttribute && !definedAsChild) {
			MIRA_THROW(XMissingParameter,
			           "Macro parameter \"" << param.name << "\" is required but missing.");
		}

		if (definedAsAttribute) {
			if (attr != node.attribute_end()) {
				arg.value.root().setName(param.name);
				arg.value.root().add_content(attr.value());
				arg.value.root().uri() = node.uri();
				arg.source.file = node.uri();
				arg.source.line = node.line();
			}
			else if (param.defaultValue) {
				arg.value = *param.defaultValue;
				arg.isDefault = true;
			}
		}
		// Reflect from tags in the call body.
		else if (param.defaultValue) {
			r.property(             //
			    param.name.c_str(), //
			    arg,                //
			    "Macro parameter.", //
			    Argument{*param.defaultValue, true, SourceInfo{}},
			    PropertyHint(), //
			    REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);
		}
		else {
			r.property(             //
			    param.name.c_str(), //
			    arg,                //
			    "Macro parameter.",
			    PropertyHint(), //
			    REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY);
		}
	}
}

} // namespace mira::xmlmacros
