/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomPreprocessor.C
 *    Implementation of XMLDomPreprocessor.h
 *
 * @author Tim Langner
 * @date   2010/10/02
 */

#include <xml/XMLDomPreprocessor.h>
#include <xml/macros/Types.h>
#include <error/Logging.h>
#include <factory/Factory.h>
#include <platform/Environment.h>
#include <utils/ResolveVariable.h>
#include <utils/PackageFinder.h>
#include <utils/PathFinder.h>
#include <utils/MakeString.h>
#include <utils/ToString.h>

#include <xml/macros/XMLMacroProcessor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::ostream& operator<<(std::ostream& s, const XMLVariableValue& x) { return s << x.value; }

///////////////////////////////////////////////////////////////////////////////

std::string resolveDefineVariable(const std::string& var,
                                  const XMLVariablesMap& variables)
{
	auto i = variables.find(var);
	if (i == variables.end())
		MIRA_THROW(XInvalidConfig, "No defined variable '" << var << "' found");
	return i->second;
}

XMLDomPreprocessor::XMLDomPreprocessor(bool expandOnly) : mMacroProcessor(std::make_unique<XMLMacroProcessor>(*this)), mExpandMacrosOnly(expandOnly) {}

XMLDomPreprocessor::XMLDomPreprocessor(const XMLVariablesMap& iVariables, bool expandOnly) :
	variables(iVariables), mMacroProcessor(std::make_unique<XMLMacroProcessor>(*this)), mExpandMacrosOnly(expandOnly) {}

XMLDomPreprocessor::~XMLDomPreprocessor() = default;

void XMLDomPreprocessor::preprocessAll(XMLDom::iterator& iNode)
{
	preprocessAllLowLevel(iNode);
}

std::string XMLDomPreprocessor::resolveContent(std::string content)
{
	if (!mExpandMacrosOnly) {
		content = resolveVariables(content, boost::bind(resolveDefineVariable, _1, variables));
		content = resolveEnvironmentVariables(content, false);
	}
	
	content = resolveSpecialVariables(content, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2), false);
	
	return content;
}

void insert(XMLDom::iterator parent,
            XMLDom::const_iterator insertBegin,
            XMLDom::const_iterator insertEnd,
            const std::string& prefixOrName,
            bool isPrefix,
            const std::vector<std::string>& filter,
            const std::string& as)
{
	for(XMLDom::const_iterator it = insertBegin; it != insertEnd; ++it)
	{
		XMLDom::iterator inserted;

		std::string nodeName = prefixOrName;

		if (isPrefix) {
			if (!nodeName.empty())
				nodeName += "/";

			nodeName += *it;
		}

		// do not insert nodes that match filter
		foreach (const std::string& f, filter) {
			if (f == nodeName)
				goto skip_node; // leave inner loop, skip to next outer loop iteration
		}

		inserted = parent.add_child(it);

		// after inserting, remove subnodes that match filter
		foreach (const std::string& f, filter) {
			size_t wildcard = f.find('*');
			if (wildcard != std::string::npos) {
				if (!(f.length() > 2 && f[f.length()-2] == '/' && wildcard == f.length()-1)) {
					MIRA_THROW(XInvalidConfig, "Invalid 'select_not' attribute for <include> tag "
					                           "(wildcard only admissible like 'path/to/subnode/*'): " << f);
				}
			}
			if ((f.size() > nodeName.size()+1) &&
			    (f.compare(0, nodeName.size()+1, nodeName+"/") == 0)) {
				if (f.length() > 2 && f[f.length()-2] == '/' && wildcard == f.length()-1) {
					// remove all child subnodes of wildcard's parent
					XMLDom::iterator removeNodeBase = parent.find(f.substr(0, f.length()-2));
					for (XMLDom::iterator it = removeNodeBase.begin(); it != removeNodeBase.end(); ) // no advance, remove() advances!
						it.remove();
				} else {
					// remove deselected subnode
					XMLDom::iterator removeNode = parent.find(f);
					removeNode.remove();
				}
			}
		}

		// after processing deselection with original names, finally rename the inserted node
		if (!as.empty())
			inserted.setName(as);

		skip_node: ;
	}
}

XMLDom::iterator XMLDomPreprocessor::parseInclude(XMLDom::iterator& ioNode)
{
	// We're only expanding macros and the include does not import any macros,
	// so it can be skipped.
	if (mExpandMacrosOnly && ioNode.find_attribute("macroNamespace") == ioNode.attribute_end()) {
		return std::next(ioNode);
	}

	try {
		Path filename(resolvePath(ioNode.get_attribute<std::string>("file"), false));
		std::string parse = ioNode.get_attribute<std::string>("parse", "xml");

		Path currentPath = Path(ioNode.uri()).parent_path();
		if (currentPath.empty() && !mIncludePathStack.empty())
			currentPath = mIncludePathStack.back().parent_path();

		if (!filename.has_root_directory())
			filename = currentPath / filename;
		filename = resolvePath(filename);

		if (parse=="xml")
		{
			if (std::find(mIncludePathStack.begin(),
			              mIncludePathStack.end(),
			              filename) != mIncludePathStack.end())
				MIRA_THROW(XInvalidConfig, "Error file '"
				           << filename.string() << "' is included recursively");
			MIRA_LOG(DEBUG) << "Including file " << filename.string();

			XMLDom xml;
			xml.loadFromFile(filename.string());
			XMLDom::iterator root = xml.root();

			mIncludePathStack.push_back(filename);
			std::string select = ioNode.get_attribute<std::string>("select", "");
			std::vector<std::string> selectedNodeNames;
			boost::split(selectedNodeNames, select, boost::is_from_range(':',':'));

			std::string selectNot = ioNode.get_attribute<std::string>("select_not", "");
			std::vector<std::string> filterNodeNames;
			boost::split(filterNodeNames, selectNot, boost::is_from_range(':',':'));

			// Use a tmp doc to hold included nodes and preprocess them
			XMLDom tmpxml;
			XMLDom::iterator tmproot = tmpxml.root();

			// if we have a select attribute add all paths of it to the tmp doc
			if (!select.empty())
			{
				std::string as = ioNode.get_attribute<std::string>("as", "");

				foreach(std::string& s, selectedNodeNames)
				{
					// '*' selects all nodes. so it is the same as no select-attribute at all,
					// with the difference that it can be selected more than once to duplicate ('*:X', '*:*')
					// (and enables the 'as' attribute)
					if (s == "*")
					{
						insert(tmproot, root.cbegin(), root.cend(), "", true, filterNodeNames, as);
					}
					// check for slash star, e.g. 'X/*' --> add all childs of X instead of the node X itself
					else if (s.length() > 2 && s[s.length()-2] == '/' && s.find('*') == s.length()-1)
					{
						s.erase(s.length()-2, 2);
						XMLDom::iterator n = root.find(s);
						insert(tmproot, n.cbegin(), n.cend(), s, true, filterNodeNames, as);
					}
					else // add node directly
					{
						if (s.find('*') != std::string::npos)
							MIRA_THROW(XInvalidConfig, "Invalid 'select' attribute for <include> tag "
							                           "(wildcard only admissible like '[path/to/subnode/]*'): " << s);
						XMLDom::iterator start = root.find(s);
						XMLDom::iterator end = start;
						++end;
						insert(tmproot, start, end, s, false, filterNodeNames, as);
					}
				}
			}
			else // no select attribute -> add all childs of root to tmp doc
			{
				insert(tmproot, root.cbegin(), root.cend(), "", true, filterNodeNames, "");
			}

			if (tmproot.cbegin() == tmproot.cend())
			{
				MIRA_LOG(WARNING) << "Including empty content from " << filename.string()
					<< " to " << ioNode.uri() <<  " (line " << ioNode.line() << ")";
			}

			mMacroProcessor->onDocumentEntered(root, tmproot);

			preprocessAll(tmproot);

			const auto targetNamespace = ioNode.get_attribute<std::string>("macroNamespace", "");
			mMacroProcessor->onDocumentLeft(targetNamespace);

			for(XMLDom::const_iterator it = tmproot.cbegin(); it != tmproot.cend(); ++it) {
				XMLDom::iterator inserted = ioNode.insert_before(it);
			}
			ioNode = ioNode.remove();
			mIncludePathStack.pop_back();
		}
		else if (parse=="text")
		{
			if (!boost::filesystem::exists(filename))
				MIRA_THROW(XFileNotFound, "The file " << filename.string() << " does not exist!");
			MIRA_LOG(DEBUG) << "Including content of file " << filename.string();
			std::ifstream ifs(filename.string().c_str());
			std::string content((std::istreambuf_iterator<char>(ifs)),
			                    (std::istreambuf_iterator<char>()));
			// remove end of line (end of file)
			if (!content.empty() && content[content.length()-1] == '\n')
				content.erase(content.length()-1);
			ioNode = ioNode.replace_by_content(resolveContent(content));
		}
		else
			MIRA_THROW(XInvalidConfig, "Unknown content for parse attribute. Must be 'xml' or 'text'")
		return ioNode;
	} catch(Exception& ex) {
		if (!ioNode.uri().empty()) {
			MIRA_RETHROW(ex, "in file included at: " <<  ioNode.uri() <<
			             " (line " << ioNode.line() << ")");
		}
		else
			throw;
	}
}

void XMLDomPreprocessor::processSpecialVariables(XMLDom::iterator& node)
{
	// Replace all occurrences of ${XXX} in attributes and content
	// with either defines or environment variables
	// Replace all occurrences of ${pattern variable} in attributes and content
	// by calling resolveXMLVariables
	XMLDom::attribute_iterator attr = node.attribute_begin();
	for( ; attr != node.attribute_end(); ++attr)
		attr = resolveContent(attr.value());
	XMLDom::content_iterator content = node.content_begin();
	for( ; content != node.content_end(); ++content)
		content = resolveContent(*content);
}

void checkCondition(XMLDom::iterator& ioNode,
                    XMLDom::iterator& ifNode,
                    XMLVariablesMap& ioVariables, bool condition)
{
	XMLDom::iterator possibleElse = ifNode; ++possibleElse;
	// if condition is met use content of <if> tag
	if (condition)
	{
		// remove if node if content is empty
		if (ifNode.begin() != ifNode.end())
		{
			XMLDom::iterator start = ifNode;
			XMLDom::iterator i = ifNode.begin();
			for (; i!=ifNode.end();++i) {
				start = start.insert_after(i);
			}
		}
		// remove else node if exists
		if (possibleElse != ioNode.end() && *possibleElse == "else")
			possibleElse.remove();
		// remove <if> node and continue
		ifNode = ifNode.remove();
	}
	else
	{
		// remove <if> node
		ifNode = ifNode.remove();
		if (possibleElse != ioNode.end() && *possibleElse == "else")
		{
			// remove else node if content is empty
			if (ifNode.begin() != ifNode.end())
			{
				XMLDom::iterator start = ifNode;
				XMLDom::iterator i = ifNode.begin();
				for (; i!=ifNode.end();++i)
					start = start.insert_after(i);
			}
			ifNode = possibleElse.remove();
		}
	}
}

bool XMLDomPreprocessor::parseCondition(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = true;
	processSpecialVariables(node);

	auto v = variables.find(it.name());
	if ((v == variables.end()) ||
	    (v->second != it.value()))
	{
		condition = false;
	}

	++it;
	if (it != node.attribute_end())
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- more than one attribute specified "
		           "(use <" << *node << "_all/any> instead)");

	return condition;
}

bool XMLDomPreprocessor::parseConditionsRequireAll(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = true;
	processSpecialVariables(node);

	for(; it != node.attribute_end(); ++it)
	{
		auto v = variables.find(it.name());
		if ((v == variables.end()) ||
		    (v->second != it.value()))
		{
			condition = false;
			break;
		}
	}

	return condition;
}

bool XMLDomPreprocessor::parseConditionsRequireAny(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();

	bool condition = false;
	processSpecialVariables(node);

	for(; it != node.attribute_end(); ++it)
	{
		auto v = variables.find(it.name());
		if ((v != variables.end()) &&
		    (v->second == it.value()))
		{
			condition = true;
			break;
		}
	}

	return condition;
}

bool XMLDomPreprocessor::parseExists(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = true;

	if (it.name() == "class")
	{
		if (!ClassFactory::isClassRegistered(it.value()))
		{
			condition = false;
		}
	}
	else if (it.name() == "file")
	{
		try
		{
			std::string value = resolveVariables(it.value(), boost::bind(resolveDefineVariable, _1, variables));
			value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
			it = value;
			Path p = resolvePath(it.value());
			if (!boost::filesystem::exists(p))
			{
				condition = false;
			}
		} catch(...)
		{
			condition = false;
		}
	}
	else if (it.name() == "var")
	{
		if (variables.count(it.value()) == 0)
		{
			condition = false;
		}
	}
	else if (it.name() == "env")
	{
		try
		{
			resolveEnvironmentVariable(it.value());
		}
		catch(...)
		{
			condition = false;
		}
	}
	else if (it.name() == "package")
	{
		try
		{
			std::string value = resolveVariables(it.value(), boost::bind(resolveDefineVariable, _1, variables));
			value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
			it = value;
			findPackage(it.value());
		} catch(...)
		{
			condition = false;
		}
	}
	else
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- unknown attribute '" << it.name() << "'");

	++it;
	if (it != node.attribute_end())
		MIRA_THROW(XInvalidConfig,
		           "Error replacing <" << *node << "> node "
		           "- more than one attribute specified "
		           " (use <" << *node << "_all/any> instead)");

	return condition;
}

bool XMLDomPreprocessor::parseExistsRequireAll(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = true;
	for(; it != node.attribute_end(); ++it)
	{
		if (it.name().rfind("class", 0) != std::string::npos)
		{
			if (!ClassFactory::isClassRegistered(it.value()))
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("file", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				Path p = resolvePath(it.value());
				if (!boost::filesystem::exists(p))
				{
					condition = false;
					break;
				}
			} catch(...)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("var", 0) != std::string::npos)
		{
			if (variables.count(it.value()) == 0)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("env", 0) != std::string::npos)
		{
			try
			{
				resolveEnvironmentVariable(it.value());
			}
			catch(...)
			{
				condition = false;
				break;
			}
		}
		else if (it.name().rfind("package", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				findPackage(it.value());
			} catch(...)
			{
				condition = false;
				break;
			}
		}
		else
			MIRA_THROW(XInvalidConfig,
			           "Error replacing <" << *node << "> node "
			           "- unknown attribute '" << it.name() << "'");
	}
	return condition;
}

bool XMLDomPreprocessor::parseExistsRequireAny(XMLDom::iterator& node)
{
	auto it = node.attribute_begin();
	bool condition = false;
	for(; it != node.attribute_end(); ++it)
	{
		if (it.name().rfind("class", 0) != std::string::npos)
		{
			if (ClassFactory::isClassRegistered(it.value()))
			{
				condition = true;
				break;
			}
		}
		else if (it.name().rfind("file", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				Path p = resolvePath(it.value());
				if (boost::filesystem::exists(p))
				{
					condition = true;
					break;
				}
			} catch(...) {}
		}
		else if (it.name().rfind("var", 0) != std::string::npos)
		{
			if (variables.count(it.value()) > 0)
			{
				condition = true;
				break;
			}
		}
		else if (it.name().rfind("env", 0) != std::string::npos)
		{
			try
			{
				resolveEnvironmentVariable(it.value());
				condition = true;
				break;
			}
			catch(...) {}
		}
		else if (it.name().rfind("package", 0) != std::string::npos)
		{
			try
			{
				std::string value = resolveVariables(it.value(),
				                                     boost::bind(resolveDefineVariable, _1, variables));
				value = resolveSpecialVariables(value, boost::bind(&XMLDomPreprocessor::resolveXMLVariables, this, _1, _2));
				it = value;
				findPackage(it.value());
				condition = true;
				break;
			} catch(...) {}
		}
		else
			MIRA_THROW(XInvalidConfig,
			           "Error replacing <" << *node << "> node "
			           "- unknown attribute '" << it.name() << "'");
	}
	return condition;
}

void XMLDomPreprocessor::preprocessAllLowLevel(XMLDom::iterator& ioNode)
{
	mAddedInfoToException = false;

	XMLVariableValue filePathBacktrack = variables["filePath"];

	XMLDom::iterator node = ioNode.begin();
	for ( ; node != ioNode.end(); )
	{
		variables["filePath"] = XMLVariableValue(Path(node.uri()).parent_path().string(),
		                                         MakeString() << node.uri() << " (line " << node.line() << ")");

		try {
			if (mMacroProcessor->isMacro(node))
			{
				node = mMacroProcessor->expand(node);
			}
			else
			if (*node == "include")
			{
				processSpecialVariables(node);
				node = parseInclude(node);
			}
			else if (*node == "define" || *node == "var")
			{
				processSpecialVariables(node);
				XMLDom::attribute_iterator ai = node.attribute_begin();
				if (ai == node.attribute_end())
					MIRA_THROW(XInvalidConfig, "Error in define/var. "
					           "No attribute specified. "
					           "Usage <define X=\"Y\"/> or <var X=\"Y\"/>");
				XMLDom::Attribute attribute = *ai;
				++ai;
				bool overwrite = false;
				for ( ; ai != node.attribute_end(); ++ai)
				{
					if (ai.name() == "overwrite") {
						overwrite = fromString<bool>(ai.value());
						break;
					}
				}
				if (variables.count(attribute.first) == 0 || overwrite)
					variables[attribute.first] = XMLVariableValue(attribute.second,
					                                              MakeString() << node.uri() << " (line " << node.line() << ")");
				// remove node
				node = node.remove();
			}
			// "expand only" mode means no if/else processing
			else if (mExpandMacrosOnly)
			{
				processSpecialVariables(node);
				preprocessAllLowLevel(node);
				++node;
			}
			else
			if (*node == "if" || *node == "if_not")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseCondition(node);
				if (*node == "if_not")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_all" || *node == "if_not_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAll(node);
				if (*node == "if_not_all")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_any" || *node == "if_not_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAny(node);
				if (*node == "if_not_any")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists" || *node == "if_not_exists")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExists(node);
				if (*node == "if_not_exists")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists_all" || *node == "if_not_exists_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAll(node);
				if (*node == "if_not_exists_all")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_exists_any" || *node == "if_not_exists_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAny(node);
				if (*node == "if_not_exists_any")
					condition = !condition;
				checkCondition(ioNode, node, variables, condition);
			}
			else
			if (*node == "if_eval")
			{
				processSpecialVariables(node);

				const auto expr = node.find_attribute("expr");

				if (expr == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					           "Error replacing <" << *node << "> node - no \"expr\" attribute specified");

				const auto rhs = node.get_attribute<std::string>("equals", "true");

				const bool rhsIsBool = boost::iequals(rhs, "true") || boost::iequals(rhs, "false");

				bool condition = false;

				if (rhsIsBool) {
					condition = fromString<bool>(expr.value()) == fromString<bool>(rhs) ;
				} else {
					condition = expr.value() == rhs;
				}

				checkCondition(ioNode, node, variables, condition);
			}
			else
			if(*node=="notice" || *node=="warning" || *node=="error" || *node=="abort")
			{
				processSpecialVariables(node);
				auto itMessage = node.find_attribute("message");
				if (itMessage == node.attribute_end())
					MIRA_THROW(XInvalidConfig, "Error in <notice>, <warning>, <error> or <abort> tag: "
					          "No message specified. Usage: <warning message=\"My message\"/>");
				if (*node =="abort")
				{
					MIRA_THROW(XInvalidConfig, "Aborting! " << itMessage.value() << " ("<<node.uri()<<":"<<node.line()<<")")
				}
				SeverityLevel lvl = WARNING;
				if(*node=="notice")
					lvl = NOTICE;
				else if(*node=="error")
					lvl = ERROR;
				MIRA_LOG(lvl) << itMessage.value() << " ("<<node.uri()<<":"<<node.line()<<")";
				node = node.remove();
			}
			else
			if (*node=="assert" || *node == "assert_not")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseCondition(node);
				if (*node == "assert_not")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_all" || *node == "assert_not_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAll(node);
				if (*node == "assert_not_all")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_any" || *node == "assert_not_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseConditionsRequireAny(node);
				if (*node == "assert_not_any")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists" || *node == "assert_not_exists")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExists(node);
				if (*node == "assert_not_exists")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists_all" || *node == "assert_not_exists_all")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAll(node);
				if (*node == "assert_not_exists_all")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node=="assert_exists_any" || *node == "assert_not_exists_any")
			{
				if (node.attribute_begin() == node.attribute_end())
					MIRA_THROW(XInvalidConfig,
					          "Error replacing <" << *node << "> node - no attribute specified");
				bool condition = parseExistsRequireAny(node);
				if (*node == "assert_not_exists_any")
					condition = !condition;
				if (!condition)
				{
					std::string message;
					if (node.content_begin() != node.content_end())
						message = *node.content_begin();
					MIRA_THROW(XInvalidConfig, "Assertion! Conditions are not met: " << message)
				}
				else // remove assert node
					node = node.remove();
			}
			else
			if (*node == "else")
			{
				MIRA_THROW(XInvalidConfig, "Error stray 'else' tag found. "
				          "(Missing 'if', 'if_not', 'if_exists' or 'if_not_exists' tag)");
			}
			else
			if (*node == "defer_resolve") {
				processSpecialVariables(node);
				auto deferIt = node.find_attribute("defer");
				bool defer = (deferIt == node.attribute_end() || fromString<bool>(deferIt.value()));
				if (defer)
				{
					// do NOT preprocess the content of this node now
					if (deferIt == node.attribute_end())
						node.add_attribute("defer", "false");
					else
						deferIt = "false"; // preprocess it next time
					++node;
				}
				else
				{
					preprocessAllLowLevel(node);
					++node;
				}
			}
			else
			{
				processSpecialVariables(node);
				preprocessAllLowLevel(node);
				++node;
			}
		} catch(Exception& ex) {
			// Macro errors hold enough information.
			if (auto xmacro = dynamic_cast<xmlmacros::XMacro*>(&ex)) {
				mAddedInfoToException = true;
			}

			if (!mAddedInfoToException && !node.uri().empty())
			{
				mAddedInfoToException = true;
				MIRA_RETHROW(ex, "while parsing tag starting at: " <<
				             node.uri() << " (line " << node.line() << ")");
			}
			else
				throw;
		}
	}

	variables["filePath"] = filePathBacktrack;
}

void XMLDomPreprocessor::preprocessXML(XMLDom& ioXml)
{
	XMLDom::iterator root = ioXml.root();
	mIncludePathStack.clear();
	mIncludePathStack.push_back(Path(ioXml.uri()));
	mMacroProcessor->onDocumentEntered(root, root);
	preprocessAll(root);
	variables.erase("filePath"); // not meaningful after processing, remove to avoid confusion
	mMacroProcessor->onDocumentLeft();
}

void XMLDomPreprocessor::registerXMLVariableCallback(const std::string& pattern, Callback&& callback)
{
	mCallbacks[pattern] = std::move(callback);
}

std::string XMLDomPreprocessor::resolveXMLVariables(const std::string& pattern, const std::string& var)
{
	std::string p = boost::to_lower_copy(pattern);
	if (p == "find")
		return findProjectPath(var);
	if (p=="findpkg")
		return findPackage(var).string();
	if (p == "env")
		return resolveEnvironmentVariable(var);
	foreach(auto cb, mCallbacks)
		if (cb.first == p)
			return cb.second(var);
	MIRA_THROW(XInvalidConfig, "Could not resolve pattern, "
			   "variable combination " << pattern << ":" << var);
	return "";
}

void preprocessXML(XMLDom& ioXml, XMLVariablesMap& variables)
{
	XMLDomPreprocessor pp(variables);
	pp.preprocessXML(ioXml);
	variables = pp.variables;
}

///////////////////////////////////////////////////////////////////////////////

}

