/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SharedLibrary.C
 *    Implementation of SharedLibrary.h
 *
 * @author Tim Langner
 * @date   2010/10/07
 */

#include <utils/SharedLibrary.h>

#include <boost/filesystem.hpp>
#include <boost/system/error_code.hpp>

#include <error/Exceptions.h>
#include <error/Logging.h>
#include <error/SystemError.h>

#include <utils/ToString.h>

#include <dlfcn.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

SharedLibrary::SharedLibrary(const Path& library) : mHandle(0)
{
	mHandle = loadSharedLibrary(library.string().c_str());
}

SharedLibrary::~SharedLibrary()
{
	if (mHandle)
		close();
}

const char* SharedLibrary::getLibPrefix()
{
	return "lib";
}

const char* SharedLibrary::getLibPostfix()
{
	return "";
}

const char* SharedLibrary::getLibExtension()
{
	return ".so";
}


Path resolveLibraryName(const Path& path)
{
	Path res = path;

	// If filename does not contain a library extension then library was
	// specified as platform independent file name without pre- and suffix.
	// hence we need to add both here
	std::string ext = SharedLibrary::getLibExtension();

	if ((res.extension() != ext) &&                       // matches libName.so
	    (res.string().find(ext+".") == std::string::npos) // matches libName.so.version
	   )
	{
		Path directory = res.parent_path();
		std::string filename = res.filename().string();
		filename = SharedLibrary::getLibPrefix()
					+ filename
					+ SharedLibrary::getLibPostfix()
					+ SharedLibrary::getLibExtension();
		res = directory / filename;
	}
	return res;
}

///////////////////////////////////////////////////////////////////////////////

SharedLibrary::Handle SharedLibrary::loadSharedLibrary(const char* library)
{
	MIRA_LOG(NOTICE) << "Loading shared library " << library;
	dlerror(); // clear any existing error
	SharedLibrary::Handle handle = dlopen(library, RTLD_NOW | RTLD_GLOBAL);
	if(handle==NULL)
		MIRA_THROW(mira::XIO, "Failed to load library '" << library << "' :" <<
		           dlerror());
	return handle;
}

SharedLibrary::Symbol SharedLibrary::getSymbol(SharedLibrary::Handle handle,
                                               const char* function)
{
	dlerror(); // clear any existing error
	SharedLibrary::Symbol symbol = dlsym(handle, function);
	if(symbol==NULL)
		MIRA_THROW(mira::XIO, "Failed to get symbol '" << function <<
		           "' from library :" << dlerror());
	return symbol;
}

void SharedLibrary::closeSharedLibrary(SharedLibrary::Handle handle)
{
	dlclose(handle);
}

///////////////////////////////////////////////////////////////////////////////

void SharedLibraryLoader::loadLibrary(const Path& library)
{
	namespace fs = boost::filesystem;
	Path fullPath = resolvePath( library );
	fullPath = resolveLibraryName( fullPath );

	if ( !fs::exists(fullPath) )
		MIRA_THROW(XFileNotFound, "File was not found: '" <<
		           fullPath.string() << "'");

	SharedLibraryPtr sl(new SharedLibrary(fullPath));
	MIRA_LOG(DEBUG) << "Successfully loaded library '" <<
		fullPath.string() << "'";
	mLibraries.push_back(sl);
}

void SharedLibraryLoader::loadLibraries(const Path& path)
{
	namespace fs = boost::filesystem;
	Path fullPath = resolvePath(path);
	if ( !fs::exists(fullPath) )
		MIRA_THROW(XIO, "Directory was not found: '" << fullPath.string() << "'");

	if ( !fs::is_directory(fullPath) )
		MIRA_THROW(XIO, "'" << fullPath.string() << "' is not a directory!");

	fs::directory_iterator end;
	for (fs::directory_iterator i(fullPath); i != end; ++i)
	{
		if ( i->path().extension() == SharedLibrary::getLibExtension() )
		{
			SharedLibraryPtr sl(new SharedLibrary(i->path()));
			MIRA_LOG(DEBUG) << "Successfully loaded library '" <<
				i->path().string() << "'";
			mLibraries.push_back(sl);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
