/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BufferStream.C
 *    Implementation of BufferStream.h
 *
 * @author Erik Einhorn
 * @date   2010/11/13
 */

#include <stream/BufferStream.h>

#include <error/Exceptions.h>

using namespace std;

namespace mira {
namespace Private {
///////////////////////////////////////////////////////////////////////////////

BufferStreamBuf::BufferStreamBuf(container_type& buffer) : mBuffer(buffer)
{
	if(mBuffer.empty())
		mBuffer.resize(1);
	setp(mBuffer.data(), mBuffer.data() + mBuffer.size());
	setg(mBuffer.data(), mBuffer.data(), mBuffer.data() + mBuffer.size());
}

int BufferStreamBuf::overflow(int c)
{
	// grow the buffer
	grow();
	// store the character that caused the overflow
	*pptr() = c;
	pbump(1); // advance put buffer
	return 0;
}

void BufferStreamBuf::grow()
{
	// current put position is the end of the buffer
	size_t ppos = pptr() - mBuffer.data();

	// current get position
	size_t gpos = gptr() - mBuffer.data();

	// grow to double size
	mBuffer.resize(mBuffer.size()*2);

	// set the put buffer
	char_type* pbegin = mBuffer.data() + ppos;
	char_type* pend   = mBuffer.data() + mBuffer.size();
	setp(pbegin, pend);

	// set the get buffer
	char_type* gbegin = mBuffer.data();
	char_type* gnext  = gbegin + gpos;
	char_type* gend   = pend;
	setg(gbegin, gnext, gend);
}

streampos BufferStreamBuf::seekoff (streamoff off, ios_base::seekdir way,
		                                 ios_base::openmode which)
{
	streampos ret = -1;
	streamoff newoffi = off;
	streamoff newoffo = off;

	char_type* begin = mBuffer.data();
	if(way == ios_base::cur) {
		newoffi += gptr() - begin;
		newoffo += pptr() - begin;
	} else if(way == ios_base::end) {
	    newoffi += egptr() - begin;
	    newoffo = newoffi;
	}

	if(which & ios_base::in) {
		gbump((begin + newoffi) - gptr());
		ret = streampos(newoffi);
	}

	if(which & ios_base::out) {
		pbump((begin + newoffo) - pptr());
		ret = streampos(newoffo);
	}
	return ret;
}


streampos BufferStreamBuf::seekpos (streampos pos, ios_base::openmode which)
{
	char_type* begin = mBuffer.data();

	if(which & ios_base::in)
		gbump((begin + pos) - gptr());

	if(which & ios_base::out)
		pbump((begin + pos) - pptr());

	return pos;
}

///////////////////////////////////////////////////////////////////////////////
}
}
