/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyNode.C
 *    Parts of implementation of PropertyNode.h.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2011/12/20
 */

#include <serialization/PropertyNode.h>

#include <boost/algorithm/string.hpp>

namespace mira {

const Duration cPropertyNodeLockTimeout = Duration::milliseconds(10);

///////////////////////////////////////////////////////////////////////////////

PropertyNode::~PropertyNode()
{
	if(mParent!=NULL) {
		int index = 0;
		for (auto it = mParent->mChildren.begin();
		          it != mParent->mChildren.end(); ++it, ++index) {
			if (*it == this) {
				mParent->removeChild(index, it);
				break;
			}
		}	
	}
	while(!mChildren.empty())
		delete *(mChildren.begin()); // will also remove child from mChildren list
}

const PropertyNode* PropertyNode::findChildNode(const std::vector<std::string>& ids,
                                                std::size_t level) const
{
	assert(level < ids.size());

	const PropertyNode* child = NULL;
	const NodeList& v = children();
	foreach(const PropertyNode* c, v)
		if(c->id()==ids[level]) {
			child = c;
			break;
		}

	if(child==NULL) // not found
		return NULL;

	// continue on next level
	++level;
	if(level == ids.size()) // reached end of id list
		return child;

	return child->findChildNode(ids,level);
}

PropertyNode* PropertyNode::findChildNode(const std::vector<std::string>& ids,
                                                std::size_t level)
{
	const PropertyNode* This = this;
	return const_cast<PropertyNode*>(This->findChildNode(ids, level));
}

const PropertyNode* PropertyNode::findChildNode(const std::string& id) const
{
	std::vector<std::string> ids;
	boost::split(ids, id, boost::is_from_range('.','.'));
	foreach (auto& id, ids)
		boost::algorithm::replace_all(id, "\\", ".");
	return findChildNode(ids,0);
}

PropertyNode* PropertyNode::findChildNode(const std::string& id)
{
	const PropertyNode* This = this;
	return const_cast<PropertyNode*>(This->findChildNode(id));
}

void PropertyNode::setFromString(const std::string& value) {
	if(isReadOnly())
		MIRA_THROW(XLogical, "Cannot set value of read-only property.");
	json::Value jsonOld = getAsJSON();
	if (jsonOld.type() == json_spirit::str_type)
		setFromJSON(json::Value(value));
	else
	{
		json::Value json;
		json::read(value, json);
		setFromJSON(json);
	}
}

std::string PropertyNode::getAsString() const {
	json::Value json = getAsJSON();
	if (json.type() == json_spirit::str_type)
		return json.get_str();
	return json::write(json, false);
}


const RootPropertyNode* PropertyNode::getRootNode() const
{
	if (!mParent)
		return NULL;

	return mParent->getRootNode();
}

RootPropertyNode* PropertyNode::getRootNode()
{
	const PropertyNode* This = this;
	return const_cast<RootPropertyNode*>(This->getRootNode());
}

void PropertyNode::addChild(PropertyNode* child, int index)
{
	if ((index < 0) || index > (int)mChildren.size())
		index = mChildren.size();

	RootPropertyNode* root = getRootNode();
	if (root) {
		// lock listeners list
		auto listeners = root->getListeners();

		beginAddChildren(*listeners, index, 1);

		mChildren.insert(std::next(mChildren.begin(), index), child);
		child->mParent = this;

		endAddChildren(*listeners);
	} else {
		mChildren.insert(std::next(mChildren.begin(), index), child);
		child->mParent = this;
	}
}

void PropertyNode::removeChild(int index, NodeList::iterator it)
{
	RootPropertyNode* root = getRootNode();
	if (root) {
		// lock listeners list
		auto listeners = root->getListeners();

		beginRemoveChildren(*listeners, index, it, 1);

		PropertyNode* node = *it;
		mChildren.erase(it);

		// this may still need the node's parent, so reset it afterwards
		endRemoveChildren(*listeners);

		node->mParent = NULL;
	} else {
		(*it)->mParent = NULL;
		mChildren.erase(it);
	}
}

void PropertyNode::removeChildren(int index, NodeList::iterator it, int count)
{
	RootPropertyNode* root = getRootNode();
	if (root) {
		// lock listeners list
		auto listeners = root->getListeners();

		beginRemoveChildren(*listeners, index, it, count);

		const NodeList::iterator& start = it;
		NodeList::iterator end = std::next(it, count);

		std::vector<PropertyNode*> nodes;
		nodes.reserve(count);
		for (auto i = start; i != end; ++i)
			nodes.push_back(*i);

		mChildren.erase(start, end);

		// this may still need the node's parents, so reset them afterwards
		endRemoveChildren(*listeners);

		foreach (auto& node, nodes)
			node->mParent = NULL;
	} else {
		const NodeList::iterator& start = it;
		NodeList::iterator end = std::next(it, count);
		for (auto i = start; i != end; ++i)
			(*i)->mParent = NULL;

		mChildren.erase(start, end);
	}
}

void PropertyNode::removeAllChildren()
{
	RootPropertyNode* root = getRootNode();
	if (root) {
		// lock listeners list
		auto listeners = root->getListeners();

		beginRemoveChildren(*listeners, 0, mChildren.begin(), mChildren.size());

		NodeList nodes;
		std::swap(nodes, mChildren); // clear mChildren, keep a copy

		// this may still need the nodes' parents, so reset them afterwards
		endRemoveChildren(*listeners);

		foreach (auto p, nodes)
			p->mParent = NULL;
	} else {
		foreach (auto p, mChildren) {
			p->mParent = NULL;
		}

		mChildren.clear();
	}
}

void PropertyNode::moveChild(int index, NodeList::iterator it, int destination, NodeList::iterator destIt)
{
	RootPropertyNode* root = getRootNode();
	if (root) {
		// lock listeners list
		auto listeners = root->getListeners();

		beginMoveChildren(*listeners, index, it, 1, destination);

		if (destination > index) {
			// i         d
			// 1,2,3,4,5,6 --> 2,3,4,5,1,6   first=i, newFirst=i+1, last=d
			std::rotate(it, std::next(it, 1), destIt);
		} else if (destination < index) {
			// d       i
			// 1,2,3,4,5,6 --> 5,1,2,3,4,6   first=d, newFirst=i, last=i+1
			std::rotate(destIt, it, std::next(it, 1));
		}
		// d==i --> no-op			

		endMoveChildren(*listeners);

	} else {
		if (destination > index)
			std::rotate(it, std::next(it, 1), destIt);
		else if (destination < index)
			std::rotate(destIt, it, std::next(it, 1));
	}
}

void PropertyNode::beginAddChildren(PropertyNodeListenerList& listeners, int index, int count)
{
	foreach(PropertyNodeListener* l, listeners)
		l->beginAddChildren(this, index, count);
}

void PropertyNode::endAddChildren(PropertyNodeListenerList& listeners)
{
	foreach(PropertyNodeListener* l, listeners)
		l->endAddChildren();
}

void PropertyNode::beginRemoveChildren(PropertyNodeListenerList& listeners, int index, NodeList::iterator it, int count)
{
	foreach(PropertyNodeListener* l,listeners)
		l->beginRemoveChildren(this, index, count);
}

void PropertyNode::endRemoveChildren(PropertyNodeListenerList& listeners)
{
	foreach(PropertyNodeListener* l, listeners)
		l->endRemoveChildren();
}

void PropertyNode::beginMoveChildren(PropertyNodeListenerList& listeners, int index, NodeList::iterator it, int count, int destination)
{
	foreach(PropertyNodeListener* l, listeners)
		l->beginMoveChildren(this, index, count, destination);
}

void PropertyNode::endMoveChildren(PropertyNodeListenerList& listeners)
{
	foreach(PropertyNodeListener* l, listeners)
		l->endMoveChildren();
}

///////////////////////////////////////////////////////////////////////////////

RootPropertyNode::RootPropertyNode() : PropertyNode("root","root", "", "", false, false)
{
}

void RootPropertyNode::addChild(PropertyNode* child, int index)
{
	PropertyNode::addChild(child, index);
}

void RootPropertyNode::removeChild(int index, PropertyNode::NodeList::iterator it)
{
	PropertyNode::removeChild(index, it);
}

void RootPropertyNode::removeChild(int index)
{
	PropertyNode::removeChild(index);
}

void RootPropertyNode::removeAllChildren()
{
	PropertyNode::removeAllChildren();
}

void RootPropertyNode::setFromJSON(const json::Value& value)
{
	MIRA_THROW(XNotImplemented, "RootPropertyNode::setFromJSON() should never be called!");
}

json::Value RootPropertyNode::getAsJSON() const
{
	MIRA_THROW(XNotImplemented, "RootPropertyNode::getAsJSON() should never be called!");
	return json::Value();
}

void RootPropertyNode::synchronize()
{
	MIRA_THROW(XNotImplemented, "RootPropertyNode::synchronize() should never be called!");
}

const RootPropertyNode* RootPropertyNode::getRootNode() const
{
	return this;
}

RootPropertyNode* RootPropertyNode::getRootNode()
{
	return this;
}

void RootPropertyNode::registerListener(PropertyNodeListener* listener)
{
	getListeners()->insert(listener);
}

void RootPropertyNode::unregisterListener(PropertyNodeListener* listener)
{
	getListeners()->erase(listener);
}

ScopedAccess<RootPropertyNode::ProtectedListenerList>
RootPropertyNode::getListeners(bool alreadyLocked)
{
	return ScopedAccess<ProtectedListenerList>(&mListeners, alreadyLocked);
}

void RootPropertyNode::lock()
{
	mMutex.lock();
}

bool RootPropertyNode::tryLock()
{
	return mMutex.timed_lock(cPropertyNodeLockTimeout);
}

void RootPropertyNode::unlock()
{
	mMutex.unlock();
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

