/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OpenSSLHelper.h
 *    Internal helper for wrapping OpenSSL
 *
 * @author Christian Martin
 * @date   2010/09/10
 */

#ifndef _MIRA_OPENSSLHELPER_H_
#define _MIRA_OPENSSLHELPER_H_

#include <openssl/aes.h>
#include <openssl/conf.h>
#include <openssl/err.h>
#include <openssl/evp.h>
#include <openssl/md5.h>
#include <openssl/objects.h>
#include <openssl/rsa.h>
#include <openssl/sha.h>
#include <openssl/x509.h>

#ifndef MIRA_USE_OPENSSL3
# include <openssl/engine.h>
#endif

#include <openssl/opensslconf.h>

#include <string>

#include <utils/Singleton.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////
// A wrapper for the OpenSSL RSA type

struct OpenSSLRSAWrapper
{
#ifdef MIRA_USE_OPENSSL3
	EVP_PKEY* key;
#else
	rsa_st* key;
#endif

	OpenSSLRSAWrapper() : key(NULL) {}
};

///////////////////////////////////////////////////////////////////////////////
// A helper class, which ensures that the OpenSSL error strings are loaded
// and freed correctly.

class OpenSSLErrorString :
	public LazySingleton<OpenSSLErrorString>
{
public:
	OpenSSLErrorString() {
#if OPENSSL_VERSION_MAJOR < 3
		ERR_load_ERR_strings();
		ERR_load_crypto_strings();
#endif
	}
	virtual ~OpenSSLErrorString() {
		ERR_free_strings();
	}

public:
	std::string err2str(unsigned long pErrCode) {
		return(std::string(ERR_error_string(pErrCode, NULL)));
	}
};

///////////////////////////////////////////////////////////////////////////////
// A helper class for cleanup OpenSSL at end of the application.
// See the following links:
//    http://www.openssl.org/support/faq.html#PROG2
//    http://www.openssl.org/support/faq.html#PROG13
//

class OpenSSLCleanup :
	public LazySingleton<OpenSSLCleanup>
{
#if OPENSSL_VERSION_NUMBER < 0x10100000L
public:
	OpenSSLCleanup() {
		CRYPTO_malloc_init();
	}

	virtual ~OpenSSLCleanup() {
		ERR_remove_state(0);
		ENGINE_cleanup();
		CONF_modules_free();
		//CONF_modules_unload(1);

		EVP_cleanup();
		CRYPTO_cleanup_all_ex_data();
	}
#endif
};

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif
