/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Process.C
 *    Implementation of Process.h.
 *
 * @author Erik Einhorn
 * @date   2011/02/10
 */

#include <platform/Process.h>
#include <error/LoggingCore.h>
#include <thread/Thread.h>

#include <stdio.h>

#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/prctl.h>

#include <string>
#include <vector>

#include <boost/algorithm/string.hpp>

#include <error/Exceptions.h>

#include "private/ProcessAncestryHelper.h"

namespace mira {

///////////////////////////////////////////////////////////////////////////////

int executeProcess(const std::string& cmd, std::string* oStdOut, std::string* oErrOut)
{
	int ret;
	if(oStdOut==NULL) {
		ret = system(cmd.c_str());
	} else {
		FILE* f = popen(cmd.c_str(), "r"); // rt

		if(f==0)
			MIRA_THROW(XRuntime, "Error while executing '" << cmd << "'");

		std::string out;
		char buffer[128];
		while(fgets(buffer, 128, f))
			out += buffer;

		*oStdOut = out;
		ret = pclose(f);
	}

	return ret;
}

///////////////////////////////////////////////////////////////////////////////

static void sigChld(int)
{
	// Do not call wait, we do the waiting ourself when shutting down.
}

Process::Process()
{
	mProcess = 0;
	// Override default signal handler for SIGCHLD, or otherwise it may
	// be waited on by the default handler.
	signal(SIGCHLD, sigChld);
}

Process::~Process()
{
	try {
		shutdown();
	} catch(Exception& ex) {
		MIRA_LOG(ERROR) << "Failed to shutdown process - " << ex.message();
	}
}

Process::Process(Process&& other) noexcept
{
	mProcess = 0;
	swap(other);
}

Process& Process::operator=(Process&& other) noexcept
{
	swap(other);
	return *this;
}

void Process::swap(Process& other)
{
	std::swap(mProcess, other.mProcess);
	std::swap(mExitCode, other.mExitCode);
	std::swap(mExitStatus, other.mExitStatus);
	std::swap(mShutdownRecursively, other.mShutdownRecursively);

	std::swap(mCinFd, other.mCinFd);
	std::swap(mCoutFd, other.mCoutFd);
	std::swap(mCerrFd, other.mCerrFd);
}

bool Process::wait(Duration maxWait)
{
	if (mProcess != 0)
	{
		Time t = Time::now();
		while(true)
		{
			int status;
			int pid = waitpid(mProcess, &status, WNOHANG | WUNTRACED);
			if (pid < 0)
				MIRA_THROW(XInvalidParameter, "Waiting for child process failed: " << strerror(errno));
			if (pid == mProcess)
			{
				if (WIFSIGNALED(status))
				{
					mExitStatus = KILLED;
					break;
				}
				if (WIFEXITED(status))
				{
					mExitCode = WEXITSTATUS(status);
					mExitStatus = NORMALEXIT;
					break;
				}
			}
			MIRA_SLEEP(10);
			if ((Time::now() - t) > maxWait)
				return false;
		}
	}
	mProcess = 0;
	return true;
}

int Process::getExitCode()
{
	if (isRunning())
		MIRA_THROW(XInvalidConfig, "Can not determine exit code of running process");
	return mExitCode;
}

Process::ExitStatus Process::getExitStatus()
{
	if (isRunning())
		MIRA_THROW(XInvalidConfig, "Can not determine exit status of running process");
	return mExitStatus;
}

bool Process::isRunning()
{
	if (mProcess == 0)
		return false;
	if (wait(Duration::seconds(0)))
		return false;
	return true;
}

void Process::interrupt()
{
	interrupt(mProcess);
}

void Process::terminate()
{
	terminate(mProcess);
}

void Process::kill()
{
	kill(mProcess);
}

void Process::setRecursiveShutdown(bool recursive)
{
	mShutdownRecursively = recursive;
}

void Process::shutdown(Duration timeout)
{
	if (mProcess == 0)
		return;

	std::vector<int> childProcs;
	if (mShutdownRecursively)
		childProcs = getRecursiveChildProcesses(mProcess, 0);

	MIRA_LOG(DEBUG) << "Waiting for process to stop...";
	if (isRunning())
	{
		MIRA_LOG(DEBUG) << "Sending process interrupt request...";
		interrupt();
		if (!wait(timeout))
		{
			MIRA_LOG(DEBUG) << "Sending process termination request...";
			terminate();
			if (!wait(timeout))
			{
				MIRA_LOG(DEBUG) << "Killing process...";
				kill();
				wait();
			}
		}
	}
	MIRA_LOG(DEBUG) << "Process stopped";

	if (mShutdownRecursively) {
		std::stringstream ss;
		foreach(int c, childProcs)
			ss << std::to_string(c) << " ";
		MIRA_LOG(DEBUG) << "Shutdown child processes: " << ss.str();
		foreach(auto c, childProcs)
			shutdown(c, timeout);
	}
}

Process Process::createProcess(const std::string& applicationName,
                               const std::vector<std::string>& args,
                               CreationFlags flags,
                               RedirectionFlags streamRedirection,
                               boost::optional<Environment> env)
{
	int statusPipeFd[2] = {0,0};
	if (pipe2(statusPipeFd, O_CLOEXEC) != 0)
		MIRA_THROW(XRuntime, "Failed to spawn process '" << applicationName << "'. Open pipes failed");

	// prepare pipes for stream redirection
	int pipeFd[3][2] = {{0,0},{0,0},{0,0}};
	if(streamRedirection & in)
		if(pipe(pipeFd[0])!=0)
			MIRA_THROW(XRuntime, "Failed to spawn process '" << applicationName << "'. Open pipes failed");
	if(streamRedirection & out)
		if(pipe(pipeFd[1])!=0)
			MIRA_THROW(XRuntime, "Failed to spawn process '" << applicationName << "'. Open pipes failed");
	if(streamRedirection & err)
		if(pipe(pipeFd[2])!=0)
			MIRA_THROW(XRuntime, "Failed to spawn process '" << applicationName << "'. Open pipes failed");

	// create new process
	pid_t pid = fork();
	if (pid == -1) // error in fork
		MIRA_THROW(XRuntime, "Failed to spawn a process '"
		           << applicationName << "'. Fork failed");

	if (pid == 0) {
		///////////////////////////////////////////////////////////////////////
		// we are the CHILD, that executes the given application

		// prepare the new process according to the flags
		// send SIGINT if parent dies: http://stackoverflow.com/questions/284325/how-to-make-child-process-die-after-parent-exits
		if(flags & interruptChildOnParentDeath)
			prctl(PR_SET_PDEATHSIG, SIGINT);

		// close unused pipe
		close(statusPipeFd[0]);

		// stream redirection
		if(streamRedirection & RedirectionFlags::in) {
			close( pipeFd[0][1] );
			dup2( pipeFd[0][0], 0); // stdin = 0
		}
		if(streamRedirection & RedirectionFlags::out) {
			close( pipeFd[1][0] );
			dup2( pipeFd[1][1], 1); // stdout = 1
		}
		if(streamRedirection & RedirectionFlags::err) {
			close( pipeFd[2][0] );
			dup2( pipeFd[2][1], 2); // stderr = 2
		}

		char** envv = NULL;
		char** argv = new char* [args.size()+2];
		argv[0] = (char*) applicationName.c_str();
		for(std::size_t i=0; i<args.size(); ++i)
			argv[i+1] = (char*) args[i].c_str();
		argv[args.size()+1]=NULL;

		if(!env) {
			execvp(applicationName.c_str(), argv);
		} else {
			// create environment variable list
			std::vector<std::string> envs = env->envp();
			envv = new char* [envs.size()+1];
			for(std::size_t i=0; i<envs.size(); ++i)
				envv[i] = (char*) envs[i].c_str();

			envv[envs.size()]=NULL;
			execvpe(applicationName.c_str(), argv, envv);
		}

		// we only reach here, if an error has occured
		delete[] envv;
		delete[] argv;

		if(write(statusPipeFd[1], &errno, sizeof(int))!=sizeof(int)) {
			MIRA_LOG(WARNING) << "Failed to write error code to parent process";
		}

		_exit(0);
	}

	///////////////////////////////////////////////////////////////////////////
	// we are the PARENT process, and pid contains the pid of the child process
	int count, err;
	close(statusPipeFd[1]);
	while ((count = read(statusPipeFd[0], &err, sizeof(errno))) == -1)
		if (errno != EAGAIN && errno != EINTR) break;
	close(statusPipeFd[0]);

	if (count) {
		// We have to wait for the process to ensure, that the (already) dead
		// process terminates. Otherwise, a <defunct> process will remain in
		// the process list.
		try {
			Process p;
			p.mProcess = pid;
			p.wait(Duration::seconds(1));
		} catch(...) {}

		if(err==2) {
			MIRA_THROW(XFileNotFound, "Failed to spawn process '"
			       << applicationName << "'. execvp failed: " << strerror(err));
		} else {
			MIRA_THROW(XInvalidParameter, "Failed to spawn process '"
			       << applicationName << "'. execvp failed: " << strerror(err));
		}
	}

	Process p;
	p.mProcess = pid;
	p.mShutdownRecursively = (flags & shutdownRecursively);

	// stream redirection
	if(streamRedirection & RedirectionFlags::in) {
		close( pipeFd[0][0] );
		p.mCinFd.reset(new ostream(boost::iostreams::file_descriptor_sink( pipeFd[0][1], boost::iostreams::close_handle )) );
	}
	if(streamRedirection & RedirectionFlags::out) {
		close( pipeFd[1][1] );
		p.mCoutFd.reset(new istream(boost::iostreams::file_descriptor_source( pipeFd[1][0], boost::iostreams::close_handle )) );
	}
	if(streamRedirection & RedirectionFlags::err) {
		close( pipeFd[2][1] );
		p.mCerrFd.reset(new istream(boost::iostreams::file_descriptor_source( pipeFd[2][0], boost::iostreams::close_handle )) );
	}

	return p;
}

uint32 Process::getPID() const
{
	return (uint32)mProcess;
}

bool Process::wait(int pid, std::vector<int>& children, Duration timeout)
{
	Time start = Time::now();
	while (true) {
		auto ancestry = priv::getProcessAncestry();
		if (!ancestry.count(pid))
			return true;

		children = priv::getChildProcesses(pid, 0, 0, ancestry);

		if (Time::now() - start > timeout)
			return false;

		MIRA_SLEEP(100);
	}
}

std::vector<int> Process::getRecursiveChildProcesses(int pid, int maxDepth)
{
	return priv::getChildProcesses(pid, 0, maxDepth, priv::getProcessAncestry());
}

bool Process::interrupt(int pid)
{
	if (pid != 0)
		return ::kill(pid, SIGINT) == 0;

	return false;
}

bool Process::terminate(int pid)
{
	if (pid != 0)
		return ::kill(pid, SIGTERM) == 0;

	return false;
}

bool Process::kill(int pid)
{
	if (pid != 0)
		return ::kill(pid, SIGKILL) == 0;

	return false;
}

void Process::shutdown(int pid, Duration timeout)
{
	auto ancestry = priv::getProcessAncestry();
	if (!ancestry.count(pid)) {
		MIRA_LOG(DEBUG) << "Process " << pid << " already quit" << std::endl;
		return;
	}

	std::vector<int> childProcs;
	childProcs = priv::getChildProcesses(pid, 0, 0, ancestry);

	MIRA_LOG(DEBUG) << "Sending process " << pid << " interrupt request...";
	bool ok = interrupt(pid);
	if (ok && !wait(pid, childProcs))
	{
		MIRA_LOG(DEBUG) << "Sending process " << pid << " termination request...";
		ok = terminate(pid);
		if (ok && !wait(pid, childProcs))
		{
			MIRA_LOG(DEBUG) << "Killing process " << pid << "...";
			ok = kill(pid);
			if (ok)
				wait(pid, childProcs);
		}
	}

	MIRA_LOG(DEBUG) << "Process " << pid << " stopped";

	if (childProcs.empty())
		return;

	std::stringstream ss;
	foreach(int c, childProcs)
		ss << std::to_string(c) << " ";
	MIRA_LOG(DEBUG) << "Shutdown child processes: " << ss.str();
	foreach(auto c, childProcs)
		shutdown(c, timeout);
}

///////////////////////////////////////////////////////////////////////////////

Process Process::createProcess(const std::string& commandLine,
                               RedirectionFlags streamRedirection)
{
	return createProcess(commandLine, noflags, streamRedirection);
}

Process Process::createProcess(const std::string& commandLine,
                               CreationFlags flags,
                               RedirectionFlags streamRedirection)
{
	std::vector<std::string> v;
	boost::split(v, commandLine, boost::algorithm::is_any_of(" \t\n"),
	             boost::algorithm::token_compress_on);
	if(v.empty())
		return Process();

	std::string application = v[0];
	std::vector<std::string> args;

	for(size_t i=1; i<v.size(); ++i)
		args.push_back(v[i]);

	return createProcess(application, args, flags, streamRedirection);
}

Process::ostream& Process::cin()
{
	return *mCinFd;
}

Process::istream& Process::cout()
{
	return *mCoutFd;
}

Process::istream& Process::cerr()
{
	return *mCerrFd;
}

///////////////////////////////////////////////////////////////////////////////
Process::Environment::Environment(const Environment& other) : mVariables(other.mVariables)
{
}

Process::Environment::Environment(Environment&& other) noexcept
{
	std::swap(mVariables,other.mVariables);
}

Process::Environment& Process::Environment::operator=(const Environment& other)
{
	mVariables = other.mVariables;
	return *this;
}

Process::Environment& Process::Environment::operator=(Environment&& other) noexcept
{
	std::swap(mVariables,other.mVariables);
	return *this;
}


void Process::Environment::clear()
{
	mVariables.clear();
}

bool Process::Environment::empty() const
{
	return mVariables.empty();
}

void Process::Environment::insert(const std::string& var, const std::string& value)
{
	mVariables[var] = value;
}

void Process::Environment::insert(const std::string& string)
{
	std::size_t sep = string.find('=');
	if(sep==std::string::npos)
		MIRA_THROW(XInvalidParameter,
		          "'" << string << "' is not a valid variable definition (use X=Y as syntax)");
	insert(string.substr(0,sep),string.substr(sep+1));
}

void Process::Environment::insertList(const std::string& string)
{
	std::vector<std::string> assigns;
	boost::split( assigns, string, boost::is_any_of(";"));
	foreach(const std::string& a, assigns)
		insert(a);
}

void Process::Environment::insert(const Environment& env)
{
	// the following 2-liner does not overwrite existing items
	//std::copy(env.mVariables.begin(), env.mVariables.end(),
	//          std::inserter(mVariables, mVariables.begin()));
	foreachIt(it, env.mVariables)
		mVariables[it->first] = it->second;
}

void Process::Environment::remove(const std::string& var)
{
	mVariables.erase(var);
}

const std::string& Process::Environment::value(const std::string& var, const std::string& defaultValue) const
{
	auto it = mVariables.find(var);
	if(it==mVariables.end())
		return defaultValue;
	return it->second;
}

Process::Environment Process::Environment::systemEnvironment()
{
	Environment env;
	for(int i=0; environ[i]!=NULL; ++i)
		env.insert(environ[i]);

	return env;
}

std::vector<std::string> Process::Environment::envp()
{
	std::vector<std::string> env;
	foreachIt(it, mVariables)
		env.push_back(it->first+"="+it->second);
	return env;
}

///////////////////////////////////////////////////////////////////////////////

}

