/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TgaFormat.C
 *  implementation of the tga format classes
 *
 * @author Erik Einhorn
 * @date   2011/05/24
 */

#include <image/TgaFormat.h>

#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

static void cfread(void *ptr, size_t size, size_t n, FILE* stream)
{
	size_t r = fread(ptr,size,n,stream);
	if(r<n)
		MIRA_THROW(XIO, "Failed to read from file");
}

struct TgaHeader
{
	// see http://www.fileformat.info/format/tga/corion.htm

	uint8 idlength;
	uint8 colormap; // 0 - no color map, 1 - 256 entry palette
	uint8 type;     // 2 - Uncompressed, RGB image, ...
	uint16 width, height;
	uint8 bpp;

	void read(FILE* file)
	{
		fseek(file, 0, SEEK_SET);
		cfread(&idlength, sizeof(uint8), 1, file);
		cfread(&colormap, sizeof(uint8), 1, file);
		cfread(&type, sizeof(uint8), 1, file);

		fseek(file, 12, SEEK_SET);
		cfread(&width, sizeof(uint16), 1, file);
		cfread(&height, sizeof(uint16), 1, file);
		cfread(&bpp, sizeof(uint8), 1, file);
	}

	void seekData(FILE* file)
	{
		fseek(file, 18+idlength, SEEK_SET);
	}

};


Img<> loadTGA(const std::string& filename)
{
	TgaHeader header;

	FILE* file = fopen(filename.c_str(), "rb");
	if(file==NULL)
		MIRA_THROW(XIO, "File not found: " << filename);

	// read header
	header.read(file);

	// check format
	if(header.colormap!=0 || header.type!=2)
		MIRA_THROW(XIO, "Invalid file: " << filename << ". Only uncompressed "
		           "TGA truecolor images are supported.");

	// read the data
	header.seekData(file);

	Img<> img;
	if(header.bpp==32)
		img = Img8U4(header.width, header.height);
	else if(header.bpp==24)
		img = Img8U3(header.width, header.height);
	else
		MIRA_THROW(XIO, "Invalid file: " << filename << ". Only 24 or 32 bit TGA "
		           "truecolor are supported.");

	std::size_t bytesperline = header.bpp / 8 * header.width;

	// read scanline by scanline
	for(uint16 y=0; y<header.height; ++y)
		cfread(img.data(header.height-y-1), 1, bytesperline, file);

	return img;
}

///////////////////////////////////////////////////////////////////////////////

}

