/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgPainter.C
 *    Implementation of ImgPainter.h.
 *
 * @author Erik Einhorn
 * @date   2012/08/16
 */

#include <image/ImgPainter.h>

#include <math/Power.h>

#include <opencv2/opencv.hpp>

namespace mira {

namespace ImgPainter {

///////////////////////////////////////////////////////////////////////////////

/**
 * Draws a line between the two given points using the specified color
 * and the specified line thickness.
 */
void drawTransparentLine(cv::Mat& ioImg, const Point2i& a, const Point2i& b,
                         const Color::RGB& color, double alpha /* =0.5f */)
{
	cv::Point p1 = a;
	cv::Point p2 = b;

	if( cv::clipLine(ioImg.size(), p1, p2) == 0)
		return; // whole line is outside of image

	cv::LineIterator iterator(ioImg, p1, p2, 4, 0);

	cv::Scalar c = color;
	for( int i = 0; i < iterator.count; i++, ++iterator)
	{
		for(int j=0; j<ioImg.channels(); ++j)
			iterator.ptr[j] = alpha*c.val[j] + (1.0f-alpha)*iterator.ptr[j];
	}
}

/**
 *
 */
void drawCovariance(cv::Mat& ioImg, const Point2i& p, float a, float b, float c,
                    float sigmascale, const Color::RGB& color, int thickness /* =1 */)
{
	float sqr = sqrt( pow2(c-a) + 4*b*b );

	float lambda1=0.5f*(c+a+sqr);
	float lambda2=0.5f*(c+a-sqr);

	// the angle of the first eigenvector
	float phi = -atan2(c-a+sqr, 2*b);
	// must be -atan, since opencv swaps the y-axis for the angle

	// must be -phi, since opencv swaps the y-axis for the angle
	phi = -phi*180.0f / pi<float>();

	cv::ellipse(ioImg, p, cv::Size2i(sqrt(lambda1)*sigmascale, sqrt(lambda2)*sigmascale),
	            phi, 0.0f, 360.0f, color, thickness);
}

/**
 *
 */
void drawInvCovariance(cv::Mat& ioImg, const Point2i& p, float a, float b, float c,
                       float sigmascale, const Color::RGB& color, int thickness /* =1 */)
{
	float sqr = sqrt( pow2(c-a) + 4*b*b );
	float det = a*c-b*b;

	// compute the eigenvals
	float lambda1=0.5f*(c+a+sqr)/det;
	float lambda2=0.5f*(c+a-sqr)/det;

	// the angle of the first eigenvector
	float phi = -atan2(c-a-sqr, 2*b);
	// must be -atan, since opencv swaps the y-axis for the angle

	// must be -phi, since opencv swaps the y-axis for the angle
	phi = -phi*180.0f / pi<float>();
	cv::ellipse(ioImg, p, cv::Size2i(sqrt(lambda1)*sigmascale, sqrt(lambda2)*sigmascale),
	            phi, 0.0f, 360.0f, color, thickness);
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace
