/*
 * Copyright (C) 2025 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Scope.h
 *    Data structure for storing macro definitions.
 *
 * @author Adrian Kriegel
 * @date   Mon Jan 20 2025
 */

#ifndef _MIRA_XML_MACROS_SCOPE_H_
#define _MIRA_XML_MACROS_SCOPE_H_

#include <string>
#include <unordered_map>

#include <xml/macros/IMacro.h>

namespace mira::xmlmacros {

struct Identifier
{
	std::string xmlns;
	std::string name;

	[[nodiscard]] std::string toString() const
	{
		return xmlns + ":" + name;
	}
};

using Definitions = std::unordered_map<std::string, xmlmacros::MacroPtr>;

/**
 * Variable scope of the macro processor.
 * Scopes work similarly to scopes in C++.
 * Each scope may contain a number of definitions.
 * Each scope may have a parent scope from which definitions are inherited.
 */
struct Scope
{
	using Ptr = std::shared_ptr<Scope>;

	Scope::Ptr parent = nullptr;

	// Set of definitions for each namespace.
	std::unordered_map<std::string, Definitions> namespaces;

	[[nodiscard]] MacroPtr findOptional(const Identifier& id) const
	{
		auto ns = namespaces.find(id.xmlns);

		if (ns != namespaces.end()) {
			auto it = ns->second.find(id.name);

			if (it != ns->second.end()) {
				return it->second;
			}
		}

		if (parent != nullptr) {
			return parent->findOptional(id);
		}

		return nullptr;
	}

	[[nodiscard]] MacroPtr find(const Identifier& name) const
	{
		auto ptr = findOptional(name);

		if (ptr == nullptr) {
			MIRA_THROW(XInvalidConfig,
			           "No definition for macro \"" + name.xmlns + ":" + name.name + "\" found.");
		}

		return ptr;
	}

	void define(Identifier id, MacroPtr macro)
	{
		auto ns = namespaces.find(id.xmlns);

		if (ns == namespaces.end()) {
			namespaces[id.xmlns] = {};
			ns = namespaces.find(id.xmlns);
		}

		if (ns->second.find(id.name) != ns->second.end()) {
			MIRA_THROW(XInvalidConfig, "Macro \"" + id.toString() + "\" already defined in this scope.");
		}

		ns->second.insert(std::make_pair(std::move(id.name), std::move(macro)));
	}

	/**
	 * Remove any definitions that no one else depends on.
	 * This may be required to avoid memory leaks in definitions which themselves hold a shared_ptr to this
	 * scope.
	 */
	void collectGarbage()
	{
		for (auto& [_, ns] : namespaces) {
			std::vector<std::remove_reference_t<decltype(ns)>::iterator> toErase;

			for (auto it = ns.begin(); it != ns.end(); ++it) {
				if (it->second.use_count() == 1) {
					toErase.emplace_back(it);
				}
			}

			for (auto it : toErase) {
				ns.erase(it);
			}
		}
	}
}; // struct Scope

using ScopePtr = Scope::Ptr;

} // namespace mira::xmlmacros

#endif // _MIRA_XML_MACROS_SCOPE_H_