/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomReflect.h
 *    Contains non-intrusive reflects for XMLDom documents.
 *    Split from XMLDom to resolve bidirectional dependency between
 *    XMLDom and XMLSerializer.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2011/01/28
 */

#ifndef _MIRA_XMLDOMREFLECT_H_
#define _MIRA_XMLDOMREFLECT_H_

#include <serialization/IsTransparentSerializable.h>
#include <serialization/SplitReflect.h>
#include <serialization/XMLSerializer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MIRA_SPLIT_REFLECT(XMLDom)

/**
 * Non-intrusive reflect for XMLDom.
 * This will (de)serialize the XMLDom to/from its string representation.
 */
template <typename Reflector>
void reflectRead(Reflector& r, XMLDom& xml)
{
	std::string s = xml.saveToString();
	r.delegate(s, REFLECT_CTRLFLAG_TEMP_TRACKING);
}

/// See the above reflect method.
template <typename Reflector>
void reflectWrite(Reflector& r, XMLDom& xml)
{
	std::string s;
	r.delegate(s, REFLECT_CTRLFLAG_TEMP_TRACKING);
	xml.loadFromString(s);
}

/**
 * Specialization for XMLSerializer.
 * This will copy XMLDom nodes directly to the XML document
 * written by the serializer.
 */
inline void reflect(XMLSerializer& r, XMLDom& xml)
{
	XMLDom::iterator node = r.getNode();
	for(XMLDom::const_iterator it = xml.root().begin(); it!=xml.root().end(); ++it)
	{
		XMLDom::iterator newnode = node.add_child("dummy");
		newnode.replace(it);
	}
}

/**
 * Specialization for XMLDeserializer.
 * This will copy XMLDom nodes directly from the XML document
 * read by the deserializer.
 */
inline void reflect(XMLDeserializer& r, XMLDom& xml)
{
	XMLDom::iterator rt = xml.root();
	// replace root to ensure complete transfer of source info (uri/line)
	rt.replace(r.getNode());
	// the name of the source node is the xml member's name in the parent object
	// rename back to 'root' for well-defined xml
	rt.setName("root");

	// we do not want to create error messages for children of r.getNode(), remove tracking for those
	r.markAllChildrenAsProcessed();
}

template <typename SerializerTag>
class IsTransparentSerializable<XMLDom, SerializerTag> : public std::true_type {};

template <>
class IsTransparentSerializable<XMLDom, XMLSerializerTag> : public std::false_type {};

///////////////////////////////////////////////////////////////////////////////

}

#endif
