/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Foreach.h
 *    Macro for iterating over all elements in a container.
 *
 * @author Erik Einhorn, Tom Mehner
 * @date   2010/05/20
 */

#ifndef _MIRA_FOREACH_H_
#define _MIRA_FOREACH_H_

#include <boost/range/adaptor/reversed.hpp>

///////////////////////////////////////////////////////////////////////////////

// Note: we do not need to enter a namespace here, since all macros are in
// the global namespace anyway

/**
 * The macro allows a PERL-or JAVA-like "foreach" construct that automates
 * the iteration over containers or other sequences that provide iterator
 * concepts.
 *
 * Examples:
 * \code
 *     std::list<int> list_int;
 *     ...
 *     foreach(int i, list_int )  // i is accessed by value here
 *     {
 *          // do something with i
 *     }
 *
 *     foreach(const int& i, list_int ) // i is accessed by const reference
 *     {                                 // here and avoids unnecessary copying.
 *          // do something with i
 *     }
 *
 *     foreach(int& i, list_int ) // i is accessed by reference here
 *     {
 *          i = ... // change the value in the container
 *     }
 *
 *     // the same also works for C-strings and std::strings:
 *     std::string str( "Hello, world!" );
 *     foreach(char ch, str)
 *     {
 *         std::cout << ch;
 *     }
 * \endcode
 *
 * Supported sequence types:
 *    - STL containers
 *    - arrays
 *    - Null-terminated strings (char and wchar_t)
 *    - std::pair of iterators
 *
 * For more details see:
 *     http://www.boost.org/doc/libs/1_43_0/doc/html/foreach.html
 */
#undef foreach // QT might have defined it already (but boost foreach is much better and compatible!!)

#define foreach(i, c) for (i : c)

/**
 * The macro can be used similar to foreach, but the containers are
 * traversed in reverse order.
 *
 * For more details see:
 *     http://www.boost.org/doc/libs/1_43_0/doc/html/foreach.html
 */
#undef foreach_reverse
#define foreach_reverse(i, c) for (i : boost::adaptors::reverse(c))

/**
 * This macro provides a more convenient way for iterating over containers or
 * other sequences that provide iterator concepts.
 * In contrast to the foreach macros above, this macro provides an iterator
 * instead of the iterated value.
 *
 * Examples:
 * \code
 *     std::list<int> list_int;
 *     ...
 *     foreachIt(it, list_int )
 *     {
 *          int val = *it;
 *     }
 * \endcode
 *
 * In fact, this macro is a simple replacement for
 * \code
 *     for(auto var=container.begin();var!=container.end();++var)
 * \endcode
 */
#define foreachIt(var, container) for (auto var = container.begin(); var != container.end(); ++var)

///////////////////////////////////////////////////////////////////////////////

#endif