/*
 * Copyright (C) 2017 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ExtensibleEnum.h
 *    A replacement for enum that can be extended in subclasses
 *
 * @author Christof Schröter
 * @date   2017/10/03
 */

#ifndef _MIRA_EXTENSIBLEENUM_H_
#define _MIRA_EXTENSIBLEENUM_H_

#include <type_traits>

#include <boost/preprocessor/tuple/to_seq.hpp>
#include <boost/preprocessor/seq/for_each.hpp>

#include <boost/thread/mutex.hpp>

#include <utils/VariadicMacro.h>

#include <serialization/SplitReflect.h>
#include <serialization/JSONSerializer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace detail {
// implementations of variants of extensible enum

///////////////////////////////////////////////////////////////////////////////

// CRTP ensures we have no common base class between derived enum classes
// -> we can detect mixup at compile time
template<typename T>
struct NoValueEnum
{
public:
	bool operator==(const NoValueEnum<T>& other) const { return ref == other.ref; }
	bool operator!=(const NoValueEnum<T>& other) const { return ref != other.ref; }

protected:
	NoValueEnum<T>() : ref(this) {}

private:
	NoValueEnum<T> const * ref;
};

///////////////////////////////////////////////////////////////////////////////

// we want one mutex per counter -> need all template parameters from
// AutoCountingEnum, even though we don't use all
template<typename T, typename CountType=int, bool UseMutex=false>
struct AutoCountingHelper
{
	static void count(CountType& i, CountType& counter)
	{
		i = counter++;
	}
};

template<typename T, typename CountType>
struct AutoCountingHelper<T, CountType, true>
{
	static void count(CountType& i, CountType& counter)
	{
		boost::mutex::scoped_lock lock(mutex);
		i = counter++;
	}

	static boost::mutex mutex;
};

template<typename T, typename CountType>
boost::mutex AutoCountingHelper<T, CountType, true>::mutex;

// CRTP additionally ensures we have one counter per direct subclass
template<typename T, typename CountType=int, bool UseMutex=false>
struct AutoCountingEnum
{
public:
	bool operator==(const AutoCountingEnum<T, CountType, UseMutex>& other) const { return i == other.i; }
	bool operator!=(const AutoCountingEnum<T, CountType, UseMutex>& other) const { return i != other.i; }

	static CountType count() { return counter; }

protected:
	AutoCountingEnum<T, CountType, UseMutex>()
	{
		AutoCountingHelper<T, CountType, UseMutex>::count(i, counter);
	}

protected:
	CountType i;
	static CountType counter;
};

template<typename T, typename CountType, bool UseMutex>
CountType AutoCountingEnum<T, CountType, UseMutex>::counter = CountType(0);

///////////////////////////////////////////////////////////////////////////////

// for Counting = false it derives from NoValueEnum
template <typename T, bool Counting=false, typename CountType=int, bool UseMutex=false>
struct ExtensibleEnumBase : public NoValueEnum<T>
{
protected:
	ExtensibleEnumBase<T, Counting, CountType, UseMutex>() {}
};

// for Counting = true it derives from AutoCountingEnum
template <typename T, typename CountType, bool UseMutex>
struct ExtensibleEnumBase<T, true, CountType, UseMutex>
	: public AutoCountingEnum<T, CountType, UseMutex>
{
protected:
	ExtensibleEnumBase<T, true, CountType, UseMutex>()
	{
		static_assert(std::is_integral<CountType>::value, "Integral required for CountType template param.");
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

///////////////////////////////////////////////////////////////////////////////

/**
 * ExtensibleEnum is a base for derived classes that can be extensible
 * 'replacements' for enum types. Like an enum, the user can define
 * a number of discrete named values in the derived class. An instance can
 * only be assigned one of these defined values.
 * The main advantage compared to an actual enum, however, is extensibility: The
 * set of valid values can easily be extended by further subclassing and
 * defining new named values in the subclass (see example below).
 *
 * Optionally each value can be assigned an additional string ID, which can e.g.
 * be used to refer to them outside of code (such as in a config file).
 * String IDs are arbitrary and not checked for uniqueness! Use of namespaces or
 * similar mechanism is recommended to keep IDs useful with extensibility.
 *
 * ExtensibleEnum can optionally use an internal counter to enumerate values
 * in the base enum class and all its derivations with internal indices,
 * in that case the total number of existing values can be queried (possibly
 * enhanced to also allow iteration over values in the future).
 * If a counter is to be used, further choices to make are the numeric type of
 * the counter and whether to use a mutex to lock the counter against parallel
 * access. All of these options are exposed as template parameters
 * (with default values). A concrete enum subclass is implemented by deriving from
 * a specific template instantiation (i.e. the choice is made by specifying base
 * class template parameters/omitting them to use defaults).
 *
 * @tparam T         The subclass name, used in CRTP
 * @tparam UseID     Whether to use a string id
 *                   (default = false)
 * @tparam Counting  bool parameter specifying whether to use an internal integral counter
 *                   (default = false)
 * @tparam CountType Type name for the integral counter type
 *                   (default = int, ignored if Counting is false)
 * @tparam UseMutex  Whether to lock the counter against parallel access,
 *                   (default = false, ignored if Counting is false)
 *
 * Macros MIRA_EXTENSIBLE_ENUM_DECLARE(EnumName, BaseClass, Values...) and
 * MIRA_EXTENSIBLE_ENUM_DEFINE(EnumName, Values...) are provided for convenience and
 * should be used.
 *
 * Usage sample code:
 * \code
 *
 * class A
 * {
 * public:
 *     MIRA_EXTENSIBLE_ENUM_DECLARE(Mode, ExtensibleEnum<Mode>,
 *                                  ModeValueA, ModeValueB)
 *
 *     void setMode(Mode mode)
 *     {
 *         mMode = mode;
 *     }
 *
 * protected:
 *
 * 	   virtual void processData(...)
 * 	   {
 * 	       if (mMode == ModeValueA)
 * 	           ...
 * 	       else
 * 	           ...
 * 	   }
 *
 * protected:
 *     Mode mMode;
 * };
 *
 * (in .C):
 * MIRA_EXTENSIBLE_ENUM_DEFINE(A::Mode, ModeValueA, ModeValueB)
 *
 * class B : public A
 * {
 * public:
 *     MIRA_EXTENSIBLE_ENUM_DECLARE(Mode, A::Mode,
 *                                  ModeValueC)
 *
 * protected:
 *    virtual void processData(...)
 * 	   {
 * 	       if (mMode == ModeValueC)
 * 	           ...
 * 	       else
 * 	           A::processInput();
 * 	   }
 * };
 *
 * (in .C):
 * MIRA_EXTENSIBLE_ENUM_DEFINE(B::Mode, ModeValueC)
 *
 * \endcode
 *
 * @note for using a multi-parameter template for the base class macro parameter,
 * it must be enclosed in parentheses (otherwise the commas will confuse the macro):
 *
 * \code
 * MIRA_EXTENSIBLE_ENUM_DECLARE(Mode, (ExtensibleEnum<Mode, true>),
 *                              ModeValueA, ModeValueB)
 *
 * For defining the IDs with each value (with UseID = true), provide them in
 * MIRA_EXTENSIBLE_ENUM_DEFINE (by default id is empty):
 *
 * \code
 *
 * MIRA_EXTENSIBLE_ENUM_DECLARE(EnumWithId, (ExtensibleEnum<Mode, true>),
 *                              ValueA, ValueB)
 *
 * MIRA_EXTENSIBLE_ENUM_DEFINE(EnumWithId, ValueA("A"), ValueB("B"))
 *
 * ....
 * if (readIDFromExternalSource() == ValueA.id())
 *     ....
 *
 * \endcode
 */

///////////////////////////////////////////////////////////////////////////////

template <typename T, bool UseID=false, bool Counting=false, typename CountType=int, bool UseMutex=false>
struct ExtensibleEnum : public detail::ExtensibleEnumBase<T, Counting, CountType, UseMutex>
{
protected:
	// we take the string param so we have a uniform interface, but ignore it
	ExtensibleEnum<T, UseID, Counting, CountType, UseMutex>(const std::string& = "") {}
};

template <typename T, bool Counting, typename CountType, bool UseMutex>
struct ExtensibleEnum<T, true, Counting, CountType, UseMutex>
	: public detail::ExtensibleEnumBase<T, Counting, CountType, UseMutex>
{
protected:
	ExtensibleEnum<T, true, Counting, CountType, UseMutex>(const std::string& id = "") : ID(id) {}

public:

	MIRA_SPLIT_REFLECT_MEMBER

	///only reflectRead(JSONSerializer&) is implemented, to be able to print()
	void reflectRead(JSONSerializer& r)
	{
		r.property("id", ID, "");
	}

	const std::string& id() const { return ID; }

private:
	std::string ID;
};

///////////////////////////////////////////////////////////////////////////////

namespace detail {

// helper to remove brackets within macro
// (which are needed to use template typenames including ',' as macro params)
template<typename> struct ExtEnum_RemoveBrackets;
template<typename T> struct ExtEnum_RemoveBrackets<void (T)> {
	typedef T Result;
};

}

#define MIRA_EXTENUM_DECLARE_VALUE(R, ENUMNAME, VALUENAME)                                    \
	const static ENUMNAME VALUENAME;

#define MIRA_EXTENSIBLE_ENUM_DECLARE(NAME, BASE, VALUES...)                                   \
	struct NAME : public mira::detail::ExtEnum_RemoveBrackets<void (BASE)>::Result            \
	{                                                                                         \
	protected:                                                                                \
		NAME(const std::string& id = "")                                                      \
			: mira::detail::ExtEnum_RemoveBrackets<void (BASE)>::Result(id) {}                \
	public:                                                                                   \
		BOOST_PP_SEQ_FOR_EACH(MIRA_EXTENUM_DECLARE_VALUE, NAME,                               \
		                      BOOST_PP_TUPLE_TO_SEQ(MIRA_VARIADIC_SIZE(VALUES), (VALUES)) )   \
	};

#define MIRA_EXTENUM_DEFINE_VALUE(R, ENUMNAME, VALUENAME)                                     \
	const ENUMNAME ENUMNAME::VALUENAME;

#define MIRA_EXTENSIBLE_ENUM_DEFINE(NAME, VALUES...)                                          \
		BOOST_PP_SEQ_FOR_EACH(MIRA_EXTENUM_DEFINE_VALUE, NAME,                                \
                              BOOST_PP_TUPLE_TO_SEQ(MIRA_VARIADIC_SIZE(VALUES), (VALUES)) )

///////////////////////////////////////////////////////////////////////////////

}

#endif
