/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Transformer.h
 *    Contains the Transformer class that contains everything to handle
 *    transformation trees and to compute transforms within those trees.
 *
 * @author Erik Einhorn
 * @date   2010/09/06
 */

#ifndef _MIRA_TRANSFORMER_H_
#define _MIRA_TRANSFORMER_H_

#include <string>
#include <list>
#include <vector>

#ifndef Q_MOC_RUN
#include <boost/thread/mutex.hpp>
#endif

#include <error/Exceptions.h>

#include <utils/Foreach.h>

#include <transform/TransformerNode.h>
#include <transform/RigidTransform.h>

#include <math/NearestNeighborInterpolator.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TransformModule
 *
 * Base class for Transformer to decouple base functionality that is
 * type independent from type dependent Transformer class template.
 */
class TransformerBase
{
public:

	/**
	 * The max. allowed number of levels within a transformation tree. It is
	 * used by collectNodesUpwardsToRoot() that is applied to find the
	 * "lowest common ancestor" in getTransformChain() to detect loops within
	 * the transformation tree.
	 */
	static const int MAX_TREE_DEPTH = 1000;

public:

	/// Injects the AbstractTransformerNode::AbstractNodePtr type
	typedef AbstractTransformerNode::AbstractNodePtr AbstractNodePtr;

public:

	/**
	 * Represents a chain or path through the transformation tree from
	 * a staring node to the target node containing nodes whose transforms
	 * need to be applied inverse and nodes whose transforms need to be applied
	 * forward.
	 */
	class Chain
	{
	public:
		std::list<AbstractNodePtr> inverse;
		std::list<AbstractNodePtr> forward;

		/// returns true, if the inverse and forward chain are both empty.
		bool empty() const { return inverse.empty() && forward.empty(); }
	};


public:

	/// The default construtor.
	TransformerBase();

	/// The destructor.
	~TransformerBase();

public:

	/**
	 * Adds a link between the specified child and its parent.
	 * Both nodes must have been added to this Transformer.
	 * If both nodes are connected to each other already this method has no
	 * effect.
	 * If the child was connected to another parent before it is unlinked from
	 * its current parent first and then linked to the new one.
	 * Returns true, if the link was new. If the link was already known, false
	 * is returned.
	 */
	bool addLink(AbstractNodePtr child, AbstractNodePtr parent);

	/**
	 * Removes an existing link between the specified child node and the
	 * given parent node. If both nodes are not connected, this method does
	 * nothing.
	 */
	void removeLink(AbstractNodePtr child, AbstractNodePtr parent);

public:

	/**
	 * Returns all nodes that are known by this transformer.
	 */
	std::list<AbstractNodePtr> getNodes();

	/**
	 * Returns all links (as parent,child-pairs) that are known by this
	 * transformer.
	 */
	std::list<std::pair<AbstractNodePtr, AbstractNodePtr>> getLinks();

	/**
	 * Returns a list of all nodes, that have no ancestor, e.g. that are root
	 * nodes of a transform tree.
	 */
	const std::list<AbstractNodePtr> getRootNodes();

public:

	/**
	 * @throw Throws XTransform, if source and target are not connected or if
	 *        there is a loop in the transform tree.
	 */
	void getTransformChain(AbstractNodePtr target, AbstractNodePtr source,
	                       Chain& oChain);

	/**
	 * Checks if a transformation between 'target' and 'source' node is available:
	 *  - Is a path between 'target' and 'source' available
	 *  - Does every Node along the path contain data
	 */
	bool isTransformAvailable(AbstractNodePtr target, AbstractNodePtr source);

public:

	/**
	 * An exception that is thrown when errors related to transformations
	 * occur.
	 */
	MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XTransform, XRuntime)

protected:

	/// Adds the specified node to the internal node map
	void addNode(AbstractNodePtr node);

	/**
	 * Returns the node with the specified id, or nullptr if such a node
	 * does not exist.
	 */
	AbstractNodePtr getNode(const std::string& nodeID);


private:

	/**
	 * Pushes all nodes starting from node up to the root node into the
	 * list 'oNodes'.
	 *
	 * @return Returns the root node that was found.
	 * @throw  Throws XTransform, if there's a loop in the transform tree.
	 */
	AbstractNodePtr collectNodesUpwardsToRoot(AbstractNodePtr node,
	                                          std::list<AbstractNodePtr>& oNodes);

protected:


	typedef std::map<std::string, AbstractNodePtr> IDToNodeMap;

	/// maps from ids to nodes that were added to us
	IDToNodeMap mNodes;

	/// protects the above node map
	boost::mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TransformModule
 * Describes a path of transformations through the transformation tree.
 * This path may consist of one or two transformation chains.
 */
struct TransformDesc
{
	TransformDesc(const TransformerBase::Chain& c1) : chain1(c1) {}

	TransformDesc(const TransformerBase::Chain& c1,
	              const TransformerBase::Chain& c2) : chain1(c1), chain2(c2) {}

	TransformerBase::Chain chain1;
	TransformerBase::Chain chain2;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TransformModule
 * A generic transformer base class that can be used with different types
 * of actual transformation nodes. The used transformation node class must
 * be derived from AbstractTransformerNode. The type of the transformation
 * nodes is passed as template parameter.
 */
template <typename TNode>
class GenericTransformer : public TransformerBase
{
	typedef TransformerBase Base;

public:

	/// The type of a transformation node.
	typedef TNode Node;

	/// The type of a pointer of the actual transformation node.
	typedef Node*       NodePtr;

	/// The type of a pointer to the AbstractTransformerNode base class
	typedef TransformerBase::AbstractNodePtr AbstractNodePtr;

public:

	/**
	 * Prepares the computation of a  transformation by computing the path
	 * between the target and source node. The returned TransformDesc can be
	 * used in a call to the getTransform() method.
	 */
	TransformDesc prepareTransform(NodePtr target, NodePtr source)
	{
		Chain chain;
		getTransformChain(target, source, chain);
		return TransformDesc(chain);
	}

	/**
	 * Prepares the computation of a  transformation by computing the path
	 * between the 'target' and 'source' node via the 'fixed' node, i.e.
	 * target->fixed->source.
	 * The returned TransformDesc can be used in a call to the getTransform()
	 * method.
	 */
	TransformDesc prepareTransform(NodePtr target, NodePtr source, NodePtr fixed)
	{
		Chain chain1, chain2;
		getTransformChain(target, fixed, chain1);
		getTransformChain(fixed, source, chain2);
		return TransformDesc(chain1, chain2);
	}

	/**
	 * Computes and returns a certain transformation that is specified via
	 * the TransformDesc. Moreover, the timestamps of the transformations is
	 * taken into account, therefore the timestamps of the target and source
	 * transforms must be specified. Additionally, a filter can be specified
	 * that can be used to interpolate or filter the transforms according to
	 * the specified timestamps.
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const TransformDesc& desc,
	                       const Time& targetTime, const Time& sourceTime,
	                       Filter&& filter)
	{
		if(desc.chain2.empty())
			return getTransform<Transform>(desc.chain1, targetTime, filter);
		else
			return getTransform<Transform>(desc.chain2, sourceTime, filter) *
			       getTransform<Transform>(desc.chain1, targetTime, filter);
	}

	/// same as above, but using NearestNeighborInterpolator as filter.
	template<typename Transform>
	Transform getTransform(const TransformDesc& desc,
	                       const Time& targetTime, const Time& sourceTime)
	{
		return getTransform<Transform>(desc, targetTime, sourceTime,
		                               NearestNeighborInterpolator());
	}


	/// same as above, but with the same time for source and target transform.
	template<typename Transform, typename Filter>
	Transform getTransform(const TransformDesc& desc, const Time& time,
			               Filter&& filter)
	{
		return getTransform<Transform>(desc, time, time, filter);
	}

	/// same as above, but using NearestNeighborInterpolator as filter.
	template<typename Transform>
	Transform getTransform(const TransformDesc& desc, const Time& time)
	{
		return getTransform<Transform>(desc, time,
		                               NearestNeighborInterpolator());
	}

	/**
	 * Computes and returns a certain transformation between the specified
	 * target and source node. Moreover, the timestamps of the transformations is
	 * taken into account. Additionally, a filter can be specified
	 * that can be used to interpolate or filter the transforms according to
	 * the specified timestamps.
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(NodePtr target, NodePtr source, const Time& time,
	                       Filter&& filter)
	{
		TransformDesc desc = prepareTransform(target, source);
		return getTransform<Transform>(desc, time, filter);
	}

	/// same as above, but using NearestNeighborInterpolator as filter
	template<typename Transform>
	Transform getTransform(NodePtr target, NodePtr source, const Time& time)
	{
		return getTransform<Transform>(target, source, time,
		                               NearestNeighborInterpolator());
	}

	/**
	 * Computes and returns a certain transformation between the specified
	 * target and source node via the 'fixed' node, i.e. target->fixed->source.
	 * Moreover, the timestamps of the transformations is
	 * taken into account, therefore the timestamps of the target and source
	 * transforms must be specified. Additionally, a filter can be specified
	 * that can be used to interpolate or filter the transforms according to
	 * the specified timestamps.
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(NodePtr target, const Time& targetTime,
	                       NodePtr source, const Time& sourceTime,
	                       NodePtr fixed, Filter&& filter)
	{
		TransformDesc desc = prepareTransform(target, source, fixed);
		return getTransform<Transform>(desc, targetTime, sourceTime, filter);
	}

	/// same as above, but using NearestNeighborInterpolator as filter
	template<typename Transform>
	Transform getTransform(NodePtr target, const Time& targetTime,
	                       NodePtr source, const Time& sourceTime,
	                       NodePtr fixed)
	{
		return getTransform<Transform>(target, targetTime,
		                               source, sourceTime, fixed,
		                               NearestNeighborInterpolator());
	}

public:

	/**
	 * Infers a transform by specifying a direct transform between two
	 * nodes, which are connected indirectly over the transformation, that shall
	 * be infered.
	 * Imagine the following example, were the transformation node 'p'
	 * is directly connected to the transformation node 'node'. Additionally,
	 * both nodes are connected via certain paths with the nodes 'source' and
	 * 'target', respectively:
	 * \verbatim
	     p ---- x ---- source
	     |
	     |
	     |
	     v
	     node --- y --- target
	   \endverbatim
	 *
	 * Now imagine, you want to compute the transformation between 'p' and
	 * 'node', but all you know is the direct transformation between
	 * the nodes 'source' and 'target' which are however not directly connected:
	 * \verbatim
	     p ---- x ---- source
	     |                .
	     | ???            . this 'transform'
	     |                . is known
	     v                v
	     node --- y --- target
	   \endverbatim
	 *
	 * In this case you can use this method. It computes and returns the
	 * desired transform on the link towards 'node' given the 'transform'
	 * between the nodes 'target' and 'source' at the specified 'time' and
	 * using the specified 'filter' for interpolation.
	 */
	template<typename Transform, typename Filter>
	Transform inferTransform(NodePtr node, NodePtr target, NodePtr source,
			                 const Transform& transform,
	                         const Time& time, Filter&& filter)
	{
		/*
		 CASE 1:
		  p ---- x ---- source
		  |               .
		  | ??            .
		  |               .
		  v               v
		 node --- y --- target
		 */

		// path from node to target
		TransformDesc node_target = prepareTransform(target,node);

		// path from node to source
		TransformDesc node_source = prepareTransform(source,node);

		// if the way from node to target goes in reverse direction towards the
		// parent of "node" (this is the case if "node" is the first one in the
		// inverse list), then we have case 2 here
		if(!node_target.chain1.inverse.empty() &&
			node_target.chain1.inverse.front()==node) {
			/*
			 CASE 2: (the way to target would be node->p->x->target,
			          via the unknown link (p->node))
			  p ---- x ---- target
			  |               ^
			  | ??            .
			  |               .
			  v               .
			 node --- y --- source
			 */

			// make sure, that the way to source does not go via the unknown
			// link (p->node), too. In this case we have a degenerate case and
			// the user has provided invalid nodes
			if(!node_source.chain1.inverse.empty() &&
				node_source.chain1.inverse.front()==node) {
				MIRA_THROW(TransformerBase::XTransform,
					"Cannot infer the transformation since all paths from '" <<
				    node->getID() << "' to source '" << source->getID() << "' "
				    "or target '" << target->getID() << "' go via "
				    "the link that should be inferred.");
			}

			Transform y = getTransform<Transform>(node_source, time, filter);

			// remove node from node->target path to get p->target path
			assert(!node_target.chain1.inverse.empty());
			assert(node_target.chain1.inverse.front()==node);
			node_target.chain1.inverse.pop_front();

			Transform x = getTransform<Transform>(node_target, time, filter);

			return x*transform.inverse()*y.inverse();

		} else {
			/*
			 CASE 1:
			 p ---- x ---- source
			  |               .
			  | ??            .
			  |               .
			  v               v
			 node --- y --- target
			 */


			Transform y = getTransform<Transform>(node_target, time, filter);

			// remove node from node->source path to get p->source path
			assert(!node_source.chain1.inverse.empty());
			assert(node_source.chain1.inverse.front()==node);
			node_source.chain1.inverse.pop_front();

			Transform x = getTransform<Transform>(node_source, time, filter);

			return x * transform * y.inverse();
		}
	}

	/// same as above, but using NearestNeighborInterpolator as filter
	template<typename Transform>
	Transform inferTransform(NodePtr node, NodePtr target, NodePtr source,
	                         const Transform& transform, const Time& time)
	{
		return inferTransform<Transform>(node, target, source, transform, time,
		                                 NearestNeighborInterpolator());
	}

public:

	/**
	 * Returns a pointer to the node with the given ID or nullptr if it does
	 * not exist.
	 */
	NodePtr getNode(const std::string& nodeID) {
		return castNode(Base::getNode(nodeID));
	}

protected:

	/// Casts an abstract node pointer to the actual used Node. This is safe.
	static NodePtr castNode(AbstractNodePtr node) {
		return static_cast<Node*>(node);
	}

	/**
	 * Computes the resulting transform by applying all transforms according
	 * to the specified transform chain.
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const Chain& chain, const Time& time, Filter&& filter)
	{
		Transform finalTransform; // start with identity transform

		// now concatenate the inverse transforms
		foreach(AbstractNodePtr n, chain.inverse)
			finalTransform*=castNode(n)->template getTransform<Transform>(time, filter).inverse();

		// concatenate the forward transforms (in reverse order because we chained them in reverse order)
		foreach_reverse(AbstractNodePtr n, chain.forward)
			finalTransform*=castNode(n)->template getTransform<Transform>(time, filter);

		return finalTransform;
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup TransformModule
 * A full features transformer class based on GenericTransformer. This
 * implementation uses instances of TransformerNode as nodes. These nodes
 * have their own data storage for storing the transformation data over time.
 * Hence, this Transformer class can be used separate from the global and
 * central application wide transform framework to generate "private" transform
 * trees and computations.
 */
template <typename T, int D, template <typename T_, int D_> class TTransform = RigidTransform>
class Transformer : public GenericTransformer<TransformerNode<TTransform<T, D>>>
{
	typedef GenericTransformer<TransformerNode<TTransform<T, D>>> Base;
public:

	/// the node type, which is a TransformerNode
	typedef TransformerNode<TTransform<T, D> > Node;

	/// pointer type to that node
	typedef Node* NodePtr;

public:

	/**
	 * Creates and adds a new node with the specified id to this transformer.
	 * @return Returns a pointer to the created node.
	 */
	NodePtr newNode(const std::string& nodeID) {
		NodePtr node(new Node(nodeID));
		this->addNode(node);
		return node;
	}
};

///////////////////////////////////////////////////////////////////////////////

/// Typedef for a transformer with 2D floating point transforms. @ingroup TransformModule
typedef Transformer<float, 2> Transformer2f;
/// Typedef for a transformer with 3D floating point transforms. @ingroup TransformModule
typedef Transformer<float, 3> Transformer3f;

/// Typedef for a transformer with 2D double precision transforms. @ingroup TransformModule
typedef Transformer<double, 2> Transformer2d;
/// Typedef for a transformer with 3D double precision transforms. @ingroup TransformModule
typedef Transformer<double, 3> Transformer3d;


/// Typedef for a transformer with 2D floating point transforms and covariance. @ingroup TransformModule
typedef Transformer<float, 2, RigidTransformCov> TransformerCov2f;
/// Typedef for a transformer with 3D floating point transforms and covariance. @ingroup TransformModule
typedef Transformer<float, 3, RigidTransformCov> TransformerCov3f;

/// Typedef for a transformer with 2D double precision transforms and covariance. @ingroup TransformModule
typedef Transformer<double, 2, RigidTransformCov> TransformerCov2d;
/// Typedef for a transformer with 3D double precision transforms and covariance. @ingroup TransformModule
typedef Transformer<double, 3, RigidTransformCov> TransformerCov3d;

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
