/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Spinlock.h
 *    Implementation of Spinlock
 *
 * @author Erik Einhorn
 * @date   2012/10/30
 */

#ifndef _MIRA_SPINLOCK_H_
#define _MIRA_SPINLOCK_H_

#ifndef Q_MOC_RUN
#include <boost/thread.hpp>
#endif
#include <thread/Atomic.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A spinlock is similar to a mutex and allows thread synchronization of
 * critical sections. However, it is more efficient than a mutex since calls
 * of kernel space methods are avoided.
 * Nevertheless, the disadvantage of a spinlock is, that it requires CPU
 * time while it is waiting to acquire the lock. It therefore should be used
 * only to protect critical sections, where not much time is spent, to reduce
 * the possible wait time.
 */
class Spinlock
{
public:

	Spinlock() {
		atomic::write(&mLocked,0);
	}

	/**
	 * Tries to lock the spinlock. Returns true, if the lock was acquired,
	 * otherwise false. This method returns immediately.
	 */
	bool try_lock()
	{
		return atomic::cas(&mLocked,1,0)==0;
	}

	/**
	 * Locks the spinlock. If it is already locked, the method blocks, until
	 * the lock can be acquired.
	 */
	void lock()
	{
		while(!try_lock());
	}

	/**
	 * Locks the spinlock. In contrast to lock(), this method relinquishes
	 * the CPU, while waiting for the lock.
	 */
	void sleeplock()
	{
		while(!try_lock())
			boost::this_thread::yield();
	}

	/**
	 * Unlocks the spinlock.
	 */
	void unlock()
	{
		atomic::write(&mLocked,0);
	}

private:

	volatile uint32 mLocked;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
