/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file variant.hpp
 *    Reflect for boost::variant
 *    This reflect requires that all the types the variant consists of are public default constructible.
 *
 * @author Tom Mehner
 * @date   Mar 10, 2023
 */

#ifndef _MIRA_BASE_INCLUDE_SERIALIZATION_ADAPTERS_BOOST_VARIANT_HPP_
#define _MIRA_BASE_INCLUDE_SERIALIZATION_ADAPTERS_BOOST_VARIANT_HPP_

#include <boost/variant.hpp>
#include <serialization/PropertySerializer.h>
#include <serialization/SplitReflect.h>
#include <tuple>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

namespace Private {

/**
 *  Helper class to call the reflector to read the content of the variant.
 *  The class provides an operator that is called by boost::apply_visitor for the type the variant contains.
 */
template<typename Reflector>
class VariantRead : public boost::static_visitor<>
{
public:
	VariantRead(Reflector& reflect) : r(reflect){};

	template<typename T>
	void operator()(T& operand) const
	{
		r.property("Content", operand, "");
	}

private:
	Reflector& r;
};

/**
 *  Helper class to call the reflector to write the content of a variant.
 *  The class provides a fill function that requires a positional argument <em>which</em>.
 *  <em>which</em> is the zero based index of the type to be written
 * (e.g. for boost::variant<int,std::string,double> int will be 0, std::string 1, double 2).
 * This class requires that every type is default constructible within its scope.
 */
template<typename Reflector, typename... Types>
class VariantWrite
{
public:
	VariantWrite(Reflector& reflect, boost::variant<Types...>& var) : r(reflect), variant(var){};

	void fill(int which)
	{
		fillInternal<Types...>(which);
	}

protected:
	/**
	 * Template function for complete list of types and all tail sublists.
	 * N is the type index for Head, determined from sizes of complete type list Types
	 * and current template parameters list Head|Tail.
	 * If N != which, the same function is called again with only Tail.
	 * The recursion will create an if/else chain to deserialize the type defined by <em>which</em>.
	 * E.g.
	 * \code
	 * if(which == 0)
	 * { // deserialize type 0 and assign to variant }
	 * else if(which == 1)
	 * { // deserialize type 1 and assign to variant }
	 * ...
	 * else
	 * { // throw }
	 * \endcode
	 */
	template<class Head, class... Tail>
	typename std::enable_if<!std::is_same<boost::detail::variant::void_, Head>::value>::type
	fillInternal(int which)
	{
		constexpr auto N = sizeof...(Types) - (sizeof...(Tail) + 1);
		if (which == N) {
			Head val;
			r.property("Content", val, "");
			variant = std::move(val);
		}
		else {
			fillInternal<Tail...>(which);
		}
	}

	/**
	 * Specialization for empty template parameter list.
	 * Current implementation of boost::variant uses specific number of template parameters,
	 * so the recursion should stop when the remaining list is empty (all available types checked).
	 */
	template<class... ARGS>
	typename std::enable_if<0 == (sizeof...(ARGS))>::type
	fillInternal(int which)
	{
		MIRA_THROW(XInvalidConfig,
		           "Invalid type index " << which << " for  " << typeName<boost::variant<Types...>>());
	}

	/**
	 * Specialization for Head == boost::detail::variant::void_.
	 * Older implementation of boost::variant has a fixed number of template parameters,
	 * where trailing unused elements are defaulted to boost::detail::variant::void_.
	 * So the recursion should stop when we encounter boost::detail::variant::void_ as Head.
	 */
	template<class Head, class... Tail>
	typename std::enable_if<std::is_same<boost::detail::variant::void_, Head>::value>::type
	fillInternal(int which)
	{
		MIRA_THROW(XInvalidConfig,
		           "Invalid type index " << which << " for  " << typeName<boost::variant<Types...>>());
	}

private:
	Reflector& r;
	boost::variant<Types...>& variant;
};

/// Create list of typenames, using recursion to iterate over variant types (see VariantWrite above)
template<typename... Types>
class TypeHintsCreator
{
public:
	static std::string create()
	{
		std::string res;
		createInternal<Types...>(res);
		return res;
	}

protected:
	template<class Head, class... Tail>
	static typename std::enable_if<!std::is_same<boost::detail::variant::void_, Head>::value>::type
	createInternal(std::string& value)
	{
		constexpr auto N = sizeof...(Types) - sizeof...(Tail) - 1;

		if (N == 0) {
			value = typeName<Head>();
		}
		else {
			value += ";" + typeName<Head>();
		}

		createInternal<Tail...>(value);
	}

	template<class... ARGS>
	static typename std::enable_if<0 == (sizeof...(ARGS))>::type
	createInternal(std::string& value)
	{}

	template<class Head, class... Tail>
	static typename std::enable_if<std::is_same<boost::detail::variant::void_, Head>::value>::type
	createInternal(std::string& value)
	{}
};

} // namespace Private

///////////////////////////////////////////////////////////////////////////////

template<typename Reflector, typename... Types>
void reflectRead(Reflector& r, boost::variant<Types...>& t)
{
	int which = t.which();
	// store which type the variant contains
	r.member("Type", which, "The actual type stored in the variant");
	// reflect the corresponding type
	boost::apply_visitor(Private::VariantRead<Reflector>{r}, t);
}

/**
 *  Specialization for PropertySerializer: 'Type' is a readonly-property
 *  with hint to show the actual typename in a PropertyEditor.
 */
template<typename... Types>
void reflectRead(PropertySerializer& r, boost::variant<Types...>& t)
{
	// store which type the variant contains
	r.roproperty("Type", getter<int>(&boost::variant<Types...>::which, &t),
	             "The actual type stored in the variant",
	             PropertyHints::enumeration(Private::TypeHintsCreator<Types...>::create()));
	// reflect the corresponding type
	boost::apply_visitor(Private::VariantRead<PropertySerializer>{r}, t);
}

template<typename Reflector, typename... Types>
void reflectWrite(Reflector& r, boost::variant<Types...>& t)
{
	int which;
	// retrieve the type the variant contains
	r.member("Type", which, "The actual type stored in the variant");
	// create and deserialize the type that corresponds to which.
	Private::VariantWrite<Reflector, Types...>{r, t}.fill(which);
}

template<typename Reflector, typename... Types>
void reflect(Reflector& r, boost::variant<Types...>& t)
{
	splitReflect(r, t);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace mira

#endif
