/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file multi_array.hpp
 *    Allows to use boost::multi_array in serialization
 *
 * @author Michael Volkhardt
 * @date   2011/01/27
 */

#ifndef _MIRA_MULTI_ARRAY_HPP_
#define _MIRA_MULTI_ARRAY_HPP_

#ifndef Q_MOC_RUN
#include <boost/array.hpp>
#include <boost/multi_array.hpp>
#endif

#include <serialization/adapters/std/StlCollections.h>
#include <serialization/IsCollection.h>

namespace mira {

////////////////////////////////////////////////////////////////////////////////

/// Specialization of the non-intrusive reflect for boost::multi_array
template<typename Reflector, typename T, std::size_t NumDims, typename Allocator>
void reflectRead(Reflector& r, boost::multi_array<T, NumDims, Allocator>& c) {

	typedef boost::multi_array<T, NumDims, Allocator> array_type;
	for (std::size_t i = 0; i < c.num_dimensions(); ++i) {
		unsigned int dim = c.shape()[i];
		MIRA_PROPERTY_WITH_ID(r, "dim", "dim[" + toString(i) + "]", dim, "");
	}

	int id = 0;
	for (auto it = c.origin(); it!= c.origin() + c.num_elements(); ++it) {
		MIRA_PROPERTY_WITH_ID(r, "item", "item[" + toString(id) + "]", *it, "");
		++id;
	}
}

/// Specialization of the non-intrusive reflect for boost::multi_array
template<typename Reflector, typename T, std::size_t NumDims, typename Allocator>
void reflectWrite(Reflector& r, boost::multi_array<T, NumDims, Allocator>& c) {

	typedef boost::multi_array<T, NumDims, Allocator> array_type;
	typedef typename array_type::index array_type_index;
	boost::array<array_type_index, NumDims> shape;

	for (std::size_t i = 0; i < c.num_dimensions(); ++i) {
		unsigned int dim;
		MIRA_PROPERTY_WITH_ID(r, "dim", "dim[" + toString(i) + "]", dim, "");
		shape[i] = dim;
	}
	c.resize(shape);

	// restore each item
	int id = 0;
	for (auto it = c.origin(); it!= c.origin() + c.num_elements(); ++it) {
		MIRA_PROPERTY_WITH_ID(r, "item", "item[" + toString(id) + "]", *it, "");
		++id;
	}
}

/// non-intrusive reflect for boost::multi_array
template<typename Reflector, typename T, std::size_t NumDims, typename Allocator>
void reflect(Reflector& r, boost::multi_array<T, NumDims, Allocator>& c) {
	splitReflect(r, c);
}

template<typename T, std::size_t NumDims, typename Allocator>
class IsCollection<boost::multi_array<T, NumDims, Allocator> > : public std::true_type {
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
