/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ReflectControlFlags.h
 *    Flags controlling reflector behaviour
 *
 * @author Christof Schröter
 * @date   2019/01/03
 */

#ifndef _MIRA_REFLECT_CONTROLFLAGS_H_
#define _MIRA_REFLECT_CONTROLFLAGS_H_

#include <utils/EnumToFlags.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Control Flags that can modify the behavior of certain reflectors.
 * Control flags can optionally be given in member()/property()/delegate() methods and affect
 * reflection of the respective element.
 */
enum ReflectCtrlFlags {
	/// No flags
	REFLECT_CTRLFLAG_NONE                 = 0x0000,

	/**
	 * This object can use object tracking internally, but the object tracking system's
	 * state remains unchanged (is restored) after the object is reflected.
	 *
	 * @note This MUST be used e.g. if function-local (non-primitive) objects are given
	 * to a member()/delegate() method, as in this case different objects may
	 * occupy the same memory location (subsequently) and otherwise erroneously
	 * be identified as address aliases.
	 *
	 * @warning Do not use function-local (or other lifetime-limited) variables/objects
	 * with property()/roproperty(), even with REFLECT_CTRLFLAG_TEMP_TRACKING!
	 * The PropertySerializer creates PropertyNodes, which hold memory references to the
	 * objects containing the property values. Expiration of a referred object's lifetime
	 * while the PropertyNode is in use will result in invalid memory access.
	 */
	REFLECT_CTRLFLAG_TEMP_TRACKING        = 0x0001,

	/**
	 * This object is volatile: its memory location may become invalid to access.
	 *
	 * Currently this flag is only relevant to property reflectors, which will use
	 * this information to avoid invalid memory access through a PropertyNode.
	 */
	REFLECT_CTRLFLAG_VOLATILE             = 0x0002,

	/**
	 * When this flag is used in calls to Reflector::member(), that member is also
	 * reflected as read-only property by instances of PropertyReflector
	 * (members are ignored by PropertyReflectors otherwise). Thus, it avoids the
	 * need for an extra call to Reflector::roproperty() to also show the member
	 * in a PropertyEditor etc.
	 *
	 * Be aware that the use as property may have other requirements than a
	 * member, e.g. see REFLECT_CTRLFLAG_VOLATILE and comments on
	 * REFLECT_CTRLFLAG_TEMP_TRACKING.
	 */
	REFLECT_CTRLFLAG_MEMBER_AS_ROPROPERTY = 0x0004

};
MIRA_ENUM_TO_FLAGS(ReflectCtrlFlags)

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
