/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IsAtomicSerializable.h
 *    Provides type trait that indicates whether a type can be serialized
 *    as atomic value.
 *
 * @author Erik Einhorn
 * @date   2010/11/01
 */

#ifndef _MIRA_ISATOMICSERIALIZABLE_H_
#define _MIRA_ISATOMICSERIALIZABLE_H_

#include <string>
#include <type_traits>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Type trait that indicates whether a type can be serialized as an atomic
 * value.
 * If this trait inherits from true_type, the type can be serialized as atomic
 * value. By default all arithmetic types (int, float, etc) can be serialized
 * as atomic value.
 * You can specialize this trait for your own types that can also be serialized
 * and deserialized as atomic values, e.g. the trait is specialized for strings
 * in this way.
 *
 * Requirement: the different serializers/deserializers will internally handle
 * atomic types by trivial/generic conversion, without needing an intrusive or
 * non-intrusive reflect() method:
 * - binary: using stream operators <</>>
 * - JSON: using json::cast()/reverse_cast() (must be overloaded for the type)
 * - XML: using toString()/fromString()
 *
 * @note: This can be used to simplify the support for binary/JSON/XML serialization,
 * but be aware that meta serialization is not fully supported and
 * binary-to-JSON conversion is not available for such types (other than those
 * for which IsAtomicSerializable is declared to be true within IsAtomicSerializable.h).
 */
template <typename T>
class IsAtomicSerializable : public std::is_fundamental<T> {};

// specialization for strings (which are treated as atomic values, too)
template<typename CharT, typename Traits, typename Alloc>
class IsAtomicSerializable<std::basic_string<CharT, Traits, Alloc>> : public std::true_type {};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_ISATOMICSERIALIZABLE_H_ */
