/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCSignature.h
 *    RPCSignature for storing all information about an RPC method signature.
 *
 * @author Erik Einhorn
 * @date   2010/11/07
 */

#ifndef _MIRA_RPCSIGNATURE_H_
#define _MIRA_RPCSIGNATURE_H_

#include <string>

#include <platform/Typename.h>
#include <serialization/adapters/std/vector>

namespace mira {

//////////////////////////////////////////////////////////////////////////////
/**
 * @ingroup RPCModule
 *
 * Stores the signature of an RPC method including the methods name and
 * its parameter types.
 */
struct RPCSignature
{
	typedef std::string ReturnType;
	typedef std::vector<std::string> ParameterTypes;

	RPCSignature() {}
	RPCSignature(const std::string& iName) : name(iName) {}
	RPCSignature(std::string&& iName) : name(std::move(iName)) {}

	bool operator<(const RPCSignature& rhs) const
	{
		int cmp = name.compare(rhs.name);
		if(cmp<0)
			return true;
		else if(cmp>0)
			return false;

		cmp = returnType.compare(rhs.returnType);
		if(cmp<0)
			return true;
		else if(cmp>0)
			return false;
		// else return types are equal, so check parameters

		if(parameterTypes.size() < rhs.parameterTypes.size())
			return true;
		else if(parameterTypes.size() > rhs.parameterTypes.size())
			return false;
		// else number of params is equal, so check the params
		// in lexicographical order
		return parameterTypes < rhs.parameterTypes;
	}

	bool operator==(const RPCSignature& rhs) const
	{
		return name == rhs.name &&
		       returnType == rhs.returnType &&
		       parameterTypes == rhs.parameterTypes;
	}

	friend std::ostream& operator<<(std::ostream& s, const RPCSignature& v)
	{
		s << v.returnType << " " <<
			v.name << "(";
		for(RPCSignature::ParameterTypes::const_iterator i=v.parameterTypes.begin();
			i!=v.parameterTypes.end(); ++i)
		{
			if(i!=v.parameterTypes.begin())
				s << ",";
			s << *i;
		}
		s << ")";
		return s;
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Name", name, "Method name");
		r.member("ReturnType", returnType, "Type of return value");
		r.member("ParameterTypes", parameterTypes, "Vector with types of parameters");
	}

	/// The method's name
	std::string name;

	/// The return type of the method
	ReturnType returnType;

	/// Vector of the type of each parameter.
	ParameterTypes parameterTypes;
};


template <typename R, typename... ARGS>
RPCSignature makeRPCSignature(std::string name) {
	RPCSignature r(std::move(name));
	r.returnType = typeName<R>();
	// the signature of an rpc consists of stripped/unqualified types
	r.parameterTypes = {typeName<typename std::remove_reference<ARGS>::type>(false)...};
	return r;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_RPCSIGNATURE_H_ */
