/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCPatternCheck.h
 *    Provides RPC SFINAE helper to check whether a provided description parameter pack is valid
 *
 * @author Tom Mehner
 * @date   Aug 18, 2023
 */

#ifndef _MIRA_BASE_INCLUDE_RPC_RPCPATTERNCHECK_H_
#define _MIRA_BASE_INCLUDE_RPC_RPCPATTERNCHECK_H_

#include <type_traits>
#include <string>

namespace mira {
namespace Private {

// Helper to detect whether the operator() exists
template<typename T>
class HasParenthesis
{
	typedef char one;
	typedef long two;

	template<typename C>
	static char test(decltype(&C::operator()));
	template<typename C>
	static long test(...);

public:
	constexpr static bool value = sizeof(test<T>(0)) == sizeof(char);
};

// checks if the underlying type of B can be constructed by A
template<typename A, typename B>
static constexpr bool isConstr()
{
	return std::is_constructible<typename std::remove_const<typename std::remove_reference<B>::type>::type,
	                             A&&>::value;
}

// type that holds a parameter pack
template<typename... ARGS>
struct ArgumentTuple
{};

// SFINAE way to detect functions and get their return values
// It is by no means complete but checks the most relevant cases
template<class F, typename Enable = void>
class FunctionTraits
{
public:
	static constexpr bool isFunction = false;
};

template<class F>
class FunctionTraits<F, typename std::enable_if<Private::HasParenthesis<F>::value>::type>
{
	using InternalParser = FunctionTraits<decltype(&F::operator())>;

public:
	using ReturnValue = typename InternalParser::ReturnValue;
	using Arguments = typename InternalParser::Arguments;

public:
	static constexpr bool isFunction = true;
};

template<class F>
struct FunctionTraits<F&> : public FunctionTraits<F>
{};

template<class F>
struct FunctionTraits<F&&> : public FunctionTraits<F>
{};

// Function Pointer
template<class R, class... Args>
struct FunctionTraits<R (*)(Args...)> : public FunctionTraits<R(Args...)>
{};

template<typename R, typename... Args>
class FunctionTraits<R(Args...)>
{
public:
	using Arguments = ArgumentTuple<Args...>;
	using ReturnValue = R;

	static constexpr bool isFunction = true;
};

// member function pointer
template<class C, class R, class... Args>
class FunctionTraits<R (C::*)(Args...)> : public FunctionTraits<R(Args...)>
{};

// const member function pointer
template<class C, class R, class... Args>
class FunctionTraits<R (C::*)(Args...) const> : public FunctionTraits<R(Args...)>
{};

// member object pointer
template<class C, class R>
class FunctionTraits<R(C::*)> : public FunctionTraits<R()>
{};

//////////////////////////////////////////////////////////////////////////////

namespace rpc {

// correctPattern()/isValid(): general compile time arguments check

template<typename... Args, typename... Descriptions>
constexpr typename std::enable_if<((sizeof...(Args)) * 3 != sizeof...(Descriptions)) &&
                                  ((sizeof...(Args)) * 2 != sizeof...(Descriptions)), bool>::type
correctPattern(ArgumentTuple<Args...> a, ArgumentTuple<Descriptions...> b)
{
	return false;
}

constexpr bool
correctPattern(ArgumentTuple<> a, ArgumentTuple<> b)
{
	return true;
}

template<typename HeadArg, typename... TailArgs, typename Name, typename Description, typename Example,
         typename... TailDocs>
constexpr typename std::enable_if<(sizeof...(TailArgs)) * 3 == sizeof...(TailDocs), bool>::type
correctPattern(ArgumentTuple<HeadArg, TailArgs...> a,
               ArgumentTuple<Name, Description, Example, TailDocs...> b)
{
	return isConstr<Example, HeadArg>() && isConstr<Name, std::string>()
	    && isConstr<Description, std::string>()
	    && correctPattern(ArgumentTuple<TailArgs...>(), ArgumentTuple<TailDocs...>());
}

template<typename HeadArg, typename... TailArgs, typename Name, typename Description,
         typename... TailDocs>
constexpr typename std::enable_if<(sizeof...(TailArgs)) * 2 == sizeof...(TailDocs), bool>::type
correctPattern(ArgumentTuple<HeadArg, TailArgs...> a,
               ArgumentTuple<Name, Description, TailDocs...> b)
{
	return isConstr<Name, std::string>() && isConstr<Description, std::string>()
	    && correctPattern(ArgumentTuple<TailArgs...>(), ArgumentTuple<TailDocs...>());
}

template<typename HeadArg, typename... TailArgs>
constexpr bool correctPattern(ArgumentTuple<HeadArg, TailArgs...> a, ArgumentTuple<> b)
{
	return true;
}

template<typename F, typename Comment, typename... Description>
constexpr typename std::enable_if<!FunctionTraits<F>::isFunction, bool>::type isValid()
{
	return false;
}

template<typename F, typename Comment, typename... Description>
constexpr typename std::enable_if<FunctionTraits<F>::isFunction, bool>::type isValid()
{
	return Private::isConstr<Comment, std::string>()
	    && correctPattern(typename FunctionTraits<F>::Arguments(), Private::ArgumentTuple<Description...>());
}

//////////////////////////////////////////////////////////////////////////////

// assertName/Description/Example(), patternError(), invalidAssertion():
// generate specific compile time error message

template<typename T>
void assertName()
{
	static_assert(
	    isConstr<T, std::string>(),
	    "_____________THE PROVIDED TYPE OF A PARAMETER NAME CAN NOT "
	    "BE USED TO INITIALIZE A STRING IN CALL TO Reflector::method(). "
	    "PLEASE PROVIDE NAME AND DESCRIPTION FOR EACH RPC METHOD PARAMETER, "
	    "OR NAME, DESCRIPTION AND A CORRECTLY TYPED SAMPLE VALUE FOR EACH PARAMETER_____________");
}

template<typename T>
void assertDescription()
{
	static_assert(
	    isConstr<T, std::string>(),
	    "_____________THE PROVIDED TYPE OF A PARAMETER DESCRIPTION CAN NOT BE USED "
	    "TO INITIALIZE A STRING IN CALL TO Reflector::method(). "
	    "PLEASE PROVIDE NAME AND DESCRIPTION FOR EACH RPC METHOD PARAMETER, "
	    "OR NAME, DESCRIPTION AND A CORRECTLY TYPED SAMPLE VALUE FOR EACH PARAMETER_____________");
}

template<typename ARG, typename EXAMPLE>
void assertExample()
{
	static_assert(
	    isConstr<EXAMPLE, ARG>(),
	    "_____________THE DEDUCED TYPE OF A PARAMETER EXAMPLE CAN NOT BE USED TO "
	    "CONSTRUCT ITS FUNCTION PARAMETER IN CALL TO Reflector::method(). "
	    "PLEASE PROVIDE NAME AND DESCRIPTION FOR EACH RPC METHOD PARAMETER, "
	    "OR NAME, DESCRIPTION AND A CORRECTLY TYPED SAMPLE VALUE FOR EACH PARAMETER_____________");
}

template<typename... Args, typename... Docs>
typename std::enable_if<((sizeof...(Args)) * 3 != sizeof...(Docs)) &&
                        ((sizeof...(Args)) * 2 != sizeof...(Docs))>::type
patternError(ArgumentTuple<Args...> a, ArgumentTuple<Docs...> b)
{
	static_assert(sizeof(ArgumentTuple<Args...>*) == 0,
	              "_____________NUMBER OF NAMES/DESCRIPTIONS/SAMPLEVALUES FOR PARAMETERS "
	              "IN CALL TO Reflector::method() DOES NOT MATCH THE PROVIDED FUNCTION'S SIGNATURE. "
	              "PLEASE PROVIDE NAME AND DESCRIPTION FOR EACH RPC METHOD PARAMETER, "
	              "OR NAME, DESCRIPTION AND A CORRECTLY TYPED SAMPLE VALUE FOR EACH PARAMETER_____________");
}

// should never happen
inline void patternError(ArgumentTuple<> a, ArgumentTuple<> b) {}

template<typename HeadArg, typename... TailArgs, typename Name, typename Description, typename Example,
         typename... TailDocs>
typename std::enable_if<(sizeof...(TailArgs)) * 3 == sizeof...(TailDocs)>::type
patternError(ArgumentTuple<HeadArg, TailArgs...> a,
             ArgumentTuple<Name, Description, Example, TailDocs...> b)
{
	assertName<Name>();
	assertDescription<Description>();
	assertExample<HeadArg, Example>();
	patternError(ArgumentTuple<TailArgs...>(), ArgumentTuple<TailDocs...>());
}

template<typename HeadArg, typename... TailArgs, typename Name, typename Description,
         typename... TailDocs>
typename std::enable_if<(sizeof...(TailArgs)) * 2 == sizeof...(TailDocs)>::type
patternError(ArgumentTuple<HeadArg, TailArgs...> a,
             ArgumentTuple<Name, Description, TailDocs...> b)
{
	assertName<Name>();
	assertDescription<Description>();
	patternError(ArgumentTuple<TailArgs...>(), ArgumentTuple<TailDocs...>());
}

template<typename F, typename Comment, typename... Description>
constexpr typename std::enable_if<!FunctionTraits<F>::isFunction>::type invalidAssertion()
{
	static_assert(sizeof(F*) == 0,
	              "_____________RECEIVED INVALID FUNCTION TYPE "
	              "IN CALL TO Reflector::method(). PLEASE PROVIDE A VALID FUNCTION OBJECT_____________");
}

template<typename F, typename Comment, typename... Description>
constexpr typename std::enable_if<FunctionTraits<F>::isFunction>::type invalidAssertion()
{
	static_assert(
	    Private::isConstr<Comment, std::string>(),
	    "_____________THE PROVIDED TYPE OF THE COMMENT CAN NOT BE USED TO INITIALIZE A "
	    "STRING IN CALL TO Reflector::method(). PLEASE PROVIDE A VALID TEXT DESCRIBING THE FUNCTION _____________");
	patternError(typename FunctionTraits<F>::Arguments(), Private::ArgumentTuple<Description...>());
}

}; // namespace rpc

//////////////////////////////////////////////////////////////////////////////

} // namespace Private


template<typename F, typename... Description>
using ValidRPCDescription = typename std::enable_if<Private::rpc::isValid<F, Description...>()>::type;

template<typename F, typename... Description>
using InvalidRPCDescription = typename std::enable_if<!Private::rpc::isValid<F, Description...>()>::type;

} // namespace mira

#endif
