/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CRSplineInterpolator.h
 *    Provides spline interpolator.
 *
 * @author Erik Einhorn
 * @date   2010/10/03
 */

#ifndef _MIRA_CRSPLINEINTERPOLATOR_H_
#define _MIRA_CRSPLINEINTERPOLATOR_H_

#include <math/IntervalFilter.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * 1D cubic Hermite spline interpolator.
 * Tangents in the control nodes are computed according to Catmull-Rom-Spline
 * (CRSpline).
 *
 * @ingroup MathModule
 */
class CRSplineInterpolator : public IntervalFilter
{
public:
	bool canExtrapolate() const { return true; }
	int samples() const         { return 4; }
	int samplesBefore() const   { return 2; }
	int samplesAfter() const    { return 2; }

	template <typename Tx, typename Ty, typename ContainerTx, typename ContainerTy>
	Ty apply(const ContainerTx& x, const ContainerTy& y, const Tx& xx)
	{
		assert(x.size()==y.size());
		assert(x.size()==4);

		const int k = 1; // this is fixed since we assume that we get exactly 4 values

		Ty mk   = (y[k+1]-y[k-1])/(x[k+1]-x[k-1]);
		Ty mkp1 = (y[k+2]-y[k])/(x[k+2]-x[k]);

		Tx scale = (x[k+1]-x[k]);

		Tx t = (xx-x[k])/scale;

		Tx t2 = t*t;
		Tx t3 = t2*t;

		// the hermite basis functions:
		Tx h00 = 2*t3-3*t2+1;
		Tx h10 = -h00+1;
		Tx h01 = t3-2*t2+t;
		Tx h11 = t3-t2;

		return h00*y[k] + h10*y[k+1] + h01*scale*mk + h11*scale*mkp1;
	}
};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_CRSPLINEINTERPOLATOR_H_ */
