/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgIterator.h
 *	ImgIterator classes that wrap OpenCV iterators.
 *
 * @author Erik Einhorn
 * @date   2011/01/27
 */

#ifndef _MIRA_IMGITERATOR_H_
#define _MIRA_IMGITERATOR_H_

#include <opencv2/core/core.hpp>

#include <geometry/Point.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/*
 * Okay, enough is enough.
 * The cv::Mat iterators were buggy in OpenCV 2.0, 2.1 and 2.2.
 * Before OpenCV 2.2 typedefs for "pointer", "reference" and
 * "iterator_category" were missing and hence the MatConstIterator_ iterator
 * was not STL-compatible.
 * In OpenCV 2.2. a couple of things were reorganized while forgetting to
 * modify some type casts accordingly, resulting in compiler errors if methods
 * like pos() were used, e.g.
 *
 * That's why we implement our own iterators (where most of the code was
 * taken from the OpenCV 2.1 implementation though)
 */
class ImgIteratorBase
{
public:

	ImgIteratorBase() : m(NULL), ptr(NULL), sliceStart(NULL), sliceEnd(NULL) {}

	ImgIteratorBase(const cv::Mat* iMat) : 
		m(iMat), ptr(NULL), sliceStart(NULL), sliceEnd(NULL)
	{
		if( m && m->isContinuous() )
		{
			sliceStart = m->data;
			sliceEnd = sliceStart + total(m)*m->elemSize();
		}
		seek(0);
	}

	ImgIteratorBase(const cv::Mat* iMat, int iRow, int iCol=0) : 
		m(iMat), ptr(NULL), sliceStart(NULL), sliceEnd(NULL)
	{
		//assert(m && m->dims <= 2);
		assert(m);
		if( m->isContinuous() )
		{
			sliceStart = m->data;
			sliceEnd = sliceStart + total(m)*m->elemSize();
		}
		ptrdiff_t ofs = iRow*m->step + iCol;
		seek(ofs);
	}

	// default copy constructor and assignment operators are fine

	/// shifts the iterator forward by the specified number of elements
	ImgIteratorBase& operator += (ptrdiff_t ofs) {
		if( !m || ofs == 0 )
			return *this;
		ptrdiff_t ofsb = ofs * m->elemSize();
		ptr += ofsb;
		if( ptr < sliceStart || sliceEnd <= ptr )
		{
			ptr -= ofsb;
			seek(ofs, true);
		}
		return *this;
	}

	/// shifts the iterator backward by the specified number of elements
	ImgIteratorBase& operator -= (ptrdiff_t ofs) {
		return (*this += -ofs);
	}

	/// move backward one element
	ImgIteratorBase& operator --() {
		if( m && (ptr -= m->elemSize()) < sliceStart )
		{
			ptr += m->elemSize();
			seek(-1, true);
		}
		return *this;
	}

	/// move forward one element
	ImgIteratorBase& operator ++() {
		if( m && (ptr += m->elemSize()) >= sliceEnd )
		{
			ptr -= m->elemSize();
			seek(1, true);
		}
		return *this;
	}

	/// returns the current iterator position
	Point2i pos() const {
		if(!m)
			return Point2i();
		//assert( m->dims <= 2 );
		ptrdiff_t ofs = ptr - m->data;
		int y = (int)(ofs/step(m));
		return Point2i((int)((ofs - y*step(m))/m->elemSize()), y);
	}

	ptrdiff_t lpos() const;

	void seek(ptrdiff_t ofs, bool relative=false);

	const cv::Mat* m;
	const uchar* ptr;
	const uchar* sliceStart;
	const uchar* sliceEnd;

protected:

	static std::size_t total(const cv::Mat* m) {
		return m->rows*m->cols;
	}

	static std::size_t step(const cv::Mat* m) {
		return m->step;
	}
};

///////////////////////////////////////////////////////////////////////////////

inline void ImgIteratorBase::seek(ptrdiff_t ofs, bool relative)
{
	assert(m!=NULL);
	std::size_t elemSize = m->elemSize();

	// empty matrix is not searchable
	if (elemSize == 0 || m->total() == 0)
		return;

	if( m->isContinuous() )
	{
		ptr = (relative ? ptr : sliceStart) + ofs*elemSize;
		if( ptr < sliceStart )
			ptr = sliceStart;
		else if( ptr > sliceEnd )
			ptr = sliceEnd;
		return;
	}

	//assert(m->dims==2);
	ptrdiff_t ofs0, y;
	if( relative )
	{
		ofs0 = ptr - m->data;
		y = ofs0/step(m);
		ofs += y*m->cols + (ofs0 - y*step(m))/elemSize;
	}
	y = ofs/m->cols;
	int y1 = std::min(std::max((int)y, 0), m->rows-1);
	sliceStart = m->data + y1*step(m);
	sliceEnd = sliceStart + m->cols*elemSize;
	ptr = y < 0 ? sliceStart : y >= m->rows ? sliceEnd :
		sliceStart + (ofs - y*m->cols)*elemSize;
}

inline ptrdiff_t ImgIteratorBase::lpos() const
{
	if(!m)
		return 0;

	std::size_t elemSize = m->elemSize();
	if( m->isContinuous() )
		return (ptr - sliceStart)/elemSize;
	ptrdiff_t ofs = ptr - m->data;

	//assert(m->dims==2);
	ptrdiff_t y = ofs/step(m);
	return y*m->cols + (ofs - y*step(m))/elemSize;
}

inline bool operator == (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return a.m == b.m && a.ptr == b.ptr; }

inline bool operator != (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return !(a == b); }

inline bool operator < (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return a.ptr < b.ptr; }

inline bool operator > (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return a.ptr > b.ptr; }

inline bool operator <= (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return a.ptr <= b.ptr; }

inline bool operator >= (const ImgIteratorBase& a, const ImgIteratorBase& b)
{ return a.ptr >= b.ptr; }

///////////////////////////////////////////////////////////////////////////////

#define MIRA_GENERATE_IMG_ITERATOR_OPS(IteratorType) \
                                                    \
IteratorType& operator+= (int ofs) {                \
	Base::operator+= (ofs);                         \
	return *this;                                   \
}                                                   \
                                                    \
IteratorType& operator-= (int ofs) {                \
	Base::operator += (-ofs);                       \
	return *this;                                   \
}                                                   \
                                                    \
IteratorType& operator--() {                        \
	Base::operator--();                             \
	return *this;                                   \
}                                                   \
                                                    \
IteratorType operator--(int){                       \
	IteratorType b = *this;                         \
	Base::operator--();                             \
	return b;                                       \
}                                                   \
                                                    \
IteratorType& operator++(){                         \
	Base::operator++();                             \
	return *this;                                   \
}                                                   \
                                                    \
IteratorType operator++(int){                       \
	IteratorType b = *this;                         \
	Base::operator++();                             \
	return b;                                       \
}

///////////////////////////////////////////////////////////////////////////////

/**
 * Const image iterator that allows to iterate over images or image regions
 * pixel by pixel similar to iterators of containers
 * @ingroup ImageModule
 */
template<typename T>
class ImgConstIterator : public ImgIteratorBase
{
	typedef ImgIteratorBase Base;

public:
	typedef T value_type;
	typedef int difference_type;

	typedef const T* pointer;
	typedef const T& reference;
	typedef std::bidirectional_iterator_tag iterator_category;

public:

	ImgConstIterator() {}
	ImgConstIterator(const cv::Mat_<T>* m) : Base(m) {}

public:

	/// Dereferences to the Pixel
	const T& operator *() const { return *(const T*)(this->ptr); }

	/// Dereferences to the Pixel
	const T* operator->() const { return (const T*)(this->ptr); }

public:
	MIRA_GENERATE_IMG_ITERATOR_OPS(ImgConstIterator)
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Image iterator that allows to iterate over images or image regions
 * pixel by pixel similar to iterators of containers
 * @ingroup ImageModule
 */
template<typename T>
class ImgIterator : public ImgConstIterator<T>
{
	typedef ImgConstIterator<T> Base;
public:

	typedef T* pointer;
	typedef T& reference;

public:

	ImgIterator() {}
	ImgIterator(cv::Mat_<T>* m) : Base(m) {}

public:

	/// Dereferences to the Pixel
	T& operator *() { return *(T*)(this->ptr); } // casts constness away, but that is safe

	/// Dereferences to the Pixel
	T* operator->() { return (T*)(this->ptr); }

public:
	MIRA_GENERATE_IMG_ITERATOR_OPS(ImgIterator)
};

///////////////////////////////////////////////////////////////////////////////

class UntypedImgIteratorValue
{
public:

	template <typename T>
	const T& at(std::size_t i) const {
		return *((T*)this + i);
	}

	template <typename T>
	T& at(std::size_t i) {
		return *((T*)this + i);
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * Untyped const image iterator, that allows to iterate over images or image
 * regions pixel by pixel similar to iterators of containers
 * @ingroup ImageModule
 */
class UntypedImgConstIterator : public ImgIteratorBase
{
	typedef ImgIteratorBase Base;
public:
	typedef UntypedImgIteratorValue value_type;
	typedef int difference_type;

	typedef const UntypedImgIteratorValue* pointer;
	typedef const UntypedImgIteratorValue& reference;
	typedef std::bidirectional_iterator_tag iterator_category;

public:

	UntypedImgConstIterator() {}
	UntypedImgConstIterator(const cv::Mat* m) : Base(m) {}

public:

	reference operator *() const { return *getValue(); }
	pointer operator->() const { return getValue(); }

public:

	MIRA_GENERATE_IMG_ITERATOR_OPS(UntypedImgConstIterator)

protected:

	value_type* getValue() { return (value_type*)(ptr); }
	const value_type* getValue() const { return (const value_type*)(ptr); }


};

///////////////////////////////////////////////////////////////////////////////


/**
 * Untyped image iterator, that allows to iterate over images or image regions
 * pixel by pixel similar to iterators of containers
 * @ingroup ImageModule
 */
class UntypedImgIterator : public UntypedImgConstIterator
{
	typedef UntypedImgConstIterator Base;
public:

	typedef UntypedImgIteratorValue* pointer;
	typedef UntypedImgIteratorValue& reference;

public:

	UntypedImgIterator() {}
	UntypedImgIterator(cv::Mat* m) : Base(m) {}

public:

	/// Dereferences to the Pixel
	reference operator *() { return *Base::getValue(); }

	/// Dereferences to the Pixel
	pointer operator->() { return Base::getValue(); }

public:

	MIRA_GENERATE_IMG_ITERATOR_OPS(UntypedImgIterator)
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
