/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Bresenham.h
 *    This header provides a set of functions to iterate over lines with the
 *    Bresenham or Bresenham Run-Slice Algorithm. It uses the visitor concept
 *    (see documentation examples) to implement the iteration.
 *
 * @author Erik Einhorn, Christof Schroeter
 * @date   2010/10/15
 */

#ifndef _MIRA_BRESENHAM_H_
#define _MIRA_BRESENHAM_H_

#include <cstdlib>
#include <algorithm>
#include <geometry/Point.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Rasterizes a Bresenham line point by point starting at coordinate (x0,y0)
 * and ending in (x1,y1).
 * For each point on the line the visitors operator()(int x, int y) is called
 * with the two parameters x,y that specify the coordinate of that point/pixel.
 *
 * Therefore, the visitor must implement the following concept:
 * \code
 * concept BresenhamVisitor
 * {
 *     void operator()(int x, int y);
 * };
 * \endcode
 *
 * @param[in] x0 x coordinate of start point
 * @param[in] y0 y coordinate of start point
 * @param[in] x1 x coordinate of end point
 * @param[in] y1 y coordinate of end point
 * @param[in,out] visitor the visitor which implements the functionality,
 *                        what to do with iterated points
 *
 * @note For optimal performance the visitor must be implemented inline!
 *
 * @see @ref GeometryPage
 */
template<typename Visitor>
void bresenham(int x0, int y0, int x1, int y1, Visitor&& visitor);

///////////////////////////////////////////////////////////////////////////////
/**
 * @ingroup GeometryModule
 *
 * Different interface for standard Bresenham Algorithm.
 * (Detailed description see @ref bresenham.)
 *
 * @param[in] p0: start point
 * @param[in] p1: end point
 * @param[in,out] visitor: the visitor which implements the functionality,
 *                         what to do with iterated points
 */
template<typename Visitor>
void bresenham(Point2i p0, Point2i p1, Visitor&& visitor)
{
	bresenham(p0[0], p0[1], p1[0], p1[1], visitor);
}

///////////////////////////////////////////////////////////////////////////////
/**
 * @ingroup GeometryModule
 *
 * Rasterizes a Bresenham line starting at coordinate (x0,y0) and
 * ending in (x1,y1). In contrast to the above method this method uses
 * the Bresenham run-slice algorithm which is significantly faster
 * for lines that mainly move along the x-axis (with small slope).
 * Instead of rastering the line pixel by pixel the line is rastered
 * segment by segment scanline by scanline from top to bottom. For
 * each segment the visitor's operator()(int x, int y, int length) is called
 * with the parameters x,y,length that specify the coordinate
 * starting position of the segment and its length.
 *
 * Therefore, the visitor must implement the following concept:
 * \code
 * concept BresenhamVisitor
 * {
 *     void operator()(int x, int y, int length);
 * };
 * \endcode
 *
 * Example:
 * \code
 *    1 2 3 4 5 6 7 8
 * 1
 * 2  x x
 * 3      x x x
 * 4            x x
 * 5
 * \endcode
 *
 * In this example the visitor would be called three times with the
 * segments (1,2,2), (3,3,3) and (6,4,2).
 *
 * Detailed information on Bresenhams Run-Slice Algorithm and source code
 * can be found here:
 *     http://www.phatcode.net/res/224/files/html/ch36/36-01.html
 * @param[in] x0: x coordinate of start point
 * @param[in] y0: y coordinate of start point
 * @param[in] x1: x coordinate of end point
 * @param[in] y1: y coordinate of end point
 * @param[in,out] visitor: the visitor which implements the functionality,
 *                         what to do with iterated points
 *
 * @note For lines with large slope (major y-axis) the standard
 *       Bresenham Algorithm is used, hence there is no advantage
 *       for those lines.
 *
 * @note For optimal performance the visitor must be implemented inline!
 *
 * @see @ref GeometryPage
 */
template<typename Visitor>
void bresenhamRunSlice(int x0, int y0, int x1, int y1, Visitor&& visitor);

///////////////////////////////////////////////////////////////////////////////
/**
 * @ingroup GeometryModule
 *
 * Different interface for Bresenham Run-Slice Algorithm.
 * Description see @ref bresenhamRunSlice.
 * @param[in] p0: start point
 * @param[in] p1: end point
 * @param[in,out] visitor: the visitor which implements the functionality,
 *                         what to do with iterated points
 */
template<typename Visitor>
void bresenhamRunSlice(Point2i p0, Point2i p1, Visitor&& visitor)
{
	bresenhamRunSlice(p0[0], p0[1], p1[0], p1[1], visitor);
}

///////////////////////////////////////////////////////////////////////////////
// implementation section: BresenhamLineIterator class definition

/**
 * @ingroup GeometryModule
 *
 * Implements an iterator that is able to iterate over a Bresenham line
 * point by point using the prefix ++operator and --operator.
 *
 * This class is generic and operates on the point coordinates only without
 * underlying data. It can be subclassed to implement a more specialized
 * iterator that is able to iterate over concrete data structures like images
 * etc. Such a subclassed iterator can be extended by a *operator and
 * ->operator to allow direct access to the data for drawing and sampling
 * lines e.g.
 *
 * @see @ref GeometryPage
 */
class BresenhamLineIterator
{
public:
	/**
	 * Creates a new bresenham iterator that iterates over a line
	 * starting at coordinate (x0,y0) and ending in (x1,y1).
	 * @param[in] x0: the x start coordinate
	 * @param[in] y0: the y start coordinate
	 * @param[in] x1: the x end coordinate
	 * @param[in] y1: the y end coordinate
	 * @param[in] xInc: the optional increment in x-direction (default = 1)
	 * @param[in] yInc: the optional increment in y-direction (default = 1)
	 */
	BresenhamLineIterator(int x0, int y0, int x1, int y1,
	                      int xInc=1, int yInc=1);

	/// Returns true, if there are more points/pixels to go.
	bool hasNext() const;

	/// Advances the iterator to the next point/pixel
	const BresenhamLineIterator& operator++();

	/// Moves iterator backward to the previous point/pixel
	const BresenhamLineIterator& operator--();

	/// Returns current x-coordinate
	int x() const { return mX; }

	/// Returns current y-coordinate
	int y() const { return mY; }

	/// Returns true if the line mainly moves along the x-axis
	bool isMajorX() const { return (mDeltaX >= mDeltaY); }

	/// Returns true if the line mainly moves along the y-axis
	bool isMajorY() const { return !isMajorX(); }

	/// Returns the current position on the line
	int pos() const	{ return mI; }

	/// Returns the length of the line in points/pixels
	int length() const { return mDen; }

	/**
	 * Creates a new bresenham line iterator that is orthogonal to this
	 * line and starts in the current point.
	 */
	BresenhamLineIterator orthogonal();

private:
	BresenhamLineIterator(int mx, int my,
	                      int xInc1, int xInc2, int yInc1, int yInc2,
	                      int num, int den, int numInc,
	                      int deltaX, int deltaY);

private:
	int mI;
	int mX,mY;
	int mXInc1, mXInc2, mYInc1, mYInc2, mNum, mDen, mNumInc;
	int mDeltaX, mDeltaY;
};

///////////////////////////////////////////////////////////////////////////////
// implementation section: GeneralBresenhamLineIterator class definition

class GeneralBresenhamLineIteratorCommonBase
{
protected:
	/**
	 * Data structure containing the numtype-independent relevant variables
	 * for one axis (dimension) of the line.
	 */
	struct AxisBase
	{
		int current;
		int step;
		int end;
		int64_t distance;
		int64_t error;
	};
};

/**
 * Guaranteeing longest axis as drive axis allows some simplification/optimization in step.
 * Since we provide a specialization for DrivenByLongestAxis=false, general version
 * is only used for the case Drive < 0 (for which it implements the automatic determination
 * of the longest axis as driving axis).
 */
template <int Drive, bool DrivenByLongestAxis = (Drive < 0)>
class GeneralBresenhamLineIteratorBase : public GeneralBresenhamLineIteratorCommonBase
{
protected:
	inline void checkForDrivingAxis(int index, const int64_t& dist, int64_t& maxd)
	{
		if (dist > maxd)
		{
			maxd = dist;
			mDrivingAxis = index;
		}
	}

	inline void step(AxisBase& axis, const AxisBase& drive)
	{
		if (axis.error >= 0)
		{
			axis.current += axis.step;
			axis.error   += axis.distance - drive.distance;
		}
		else
			axis.error += axis.distance;
	}

public:
	/// Returns dimension index of the driving axis.
	inline int drivingAxis() const { return mDrivingAxis; }

protected:
	int mDrivingAxis;    ///< driving axis dimension index
};

/**
 * Specialization for DrivenByLongestAxis=false (driving axis given explicitly:
 * No need to determine the drive axis, but stepping is a little more complex.
 */
template <int Drive>
class GeneralBresenhamLineIteratorBase<Drive, false> : public GeneralBresenhamLineIteratorCommonBase
{
protected:
	inline void checkForDrivingAxis(int index, const int64_t& dist, int64_t& maxd) {} // do nothing

	inline void step(AxisBase& axis, const AxisBase& drive)
	{
		if (axis.error >= 0)
		{
			int steps = axis.error / drive.distance + 1;
			axis.current += steps * axis.step;
			axis.error   -= steps * drive.distance;
		}
		axis.error += axis.distance;
	}

public:
	/// Returns dimension index of the driving axis.
	inline int drivingAxis() const { return Drive; }
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * This more general iterator basically follows the design of
 * @ref BresenhamLineIterator, but works on an arbitrary-dimensional line and
 * handles real-valued start and end positions.
 *
 * @tparam D: The dimension of the space the line is defined in.
 * @tparam T: The numerical type of the  start/end point values (default: float).
 * @tparam Drive: The (dimension index of the) driving axis. The iterator hits each
 *                point/pixel between start and end on the driving axis exactly
 *                once. For a dense line, the driving axis must be the longest
 *                axis. This longest axis will be determined as the driving axis
 *                automatically if Drive is set < 0 (the default).
 *                Guaranteeing the longest axis as driving axis also enables some
 *                (compile-time) optimization for the iteration, so use with Drive=-1
 *                is encouraged.
 * @tparam Res: Resolution (default = 1000). Real-valued increments are
 *              translated to integer values (to allow for integer-arithmetic
 *              calculations) by multiplying with Res.
 *
 * for details: http://www.idav.ucdavis.edu/education/GraphicsNotes/Bresenhams-Algorithm/Bresenhams-Algorithm.html
 *
 * @see @ref GeometryPage
 */
template <int D, typename T = float, int Drive = -1, int Res = 1000>
class GeneralBresenhamLineIterator : public GeneralBresenhamLineIteratorBase<Drive>
{
	static_assert(Drive < D, "Driving axis Drive must not be >= number of dimensions D in GeneralBresenhamLineIterator instantiation!");
public:
	/**
	 * Add the type-dependent start and end value to the axis data struct.
	 */
	struct Axis : public GeneralBresenhamLineIteratorCommonBase::AxisBase
	{
		T startExact, endExact;
	};

public:
	/**
	 * Creates a new bresenham iterator that iterates over a line
	 * starting at point start and ending in end.
	 * @param[in] start: the D-dimensional real-valued start point
	 * @param[in] end: the D-dimensional real-valued end point
	 */
	GeneralBresenhamLineIterator(Point<T, D> start = Point<T, D>(),
	                             Point<T, D> end = Point<T, D>());

public:
	/**
	 * Set a new start and end, reset iterator.
	 * @param[in] start: the D-dimensional real-valued start point
	 * @param[in] end: the D-dimensional real-valued end point
	 */
	void init(Point<T, D> start = Point<T, D>(),
	          Point<T, D> end = Point<T, D>());

	/// Returns true, if there are more points/pixels to go.
	inline bool hasNext() const
		{ return mAxes[this->drivingAxis()].current != mAxes[this->drivingAxis()].end; }

	/// Advances the iterator to the next point/pixel.
	const GeneralBresenhamLineIterator& operator++();

	/**
	 * Returns the current position (in all dimensions).
	 * This method is just for convenience and is slow (as it creates the
	 * returned point variable from the internal axes variables). If you care
	 * about performance, query the position for each dimension using
	 * @ref axis().
	 */
	Point<int, D> pos() const;

	/**
	 * Returns axis structure for a specified dimension.
	 * @param[in] d: dimension, must be < D.
	 */
	inline const Axis& axis(uint32 d) const { return mAxes[d]; }

protected:

	Axis mAxes[D];       ///< the axes variables
};

///////////////////////////////////////////////////////////////////////////////
// implementation section: BresenhamLineIterator class implementation
///////////////////////////////////////////////////////////////////////////////

inline BresenhamLineIterator::BresenhamLineIterator(int x0, int y0,
                                                    int x1, int y1,
                                                    int xInc, int yInc)
{
	// prepare the bresenham algo
	mDeltaX = std::abs(x1 - x0);
	mDeltaY = std::abs(y1 - y0);

	mX = x0;
	mY = y0;

	if (x1 >= x0) {mXInc1 =  xInc; mXInc2 =  xInc; }
			 else {mXInc1 = -xInc; mXInc2 = -xInc; }

	if (y1 >= y0) {mYInc1 =  yInc; mYInc2 =  yInc; }
			 else {mYInc1 = -yInc; mYInc2 = -yInc; }

	if (mDeltaX >= mDeltaY) {
		mXInc1  = 0;
		mYInc2  = 0;
		mDen    = mDeltaX;
		mNumInc = mDeltaY;
	}
	else {
		mXInc2  = 0;
		mYInc1  = 0;
		mDen    = mDeltaY;
		mNumInc = mDeltaX;
	}

	mNum = mDen >> 1;
	mI=0;
}

///////////////////////////////////////////////////////////////////////////////

inline BresenhamLineIterator::BresenhamLineIterator(int mx, int my,
                                                    int xInc1, int xInc2,
                                                    int yInc1, int yInc2,
                                                    int num, int den, int numInc,
                                                    int deltaX, int deltaY) :
	mI(0),
	mX(mx), mY(my),
	mXInc1(xInc1), mXInc2(xInc2), mYInc1(yInc1), mYInc2(yInc2),
	mNum(num), mDen(den), mNumInc(numInc),
	mDeltaX(deltaX), mDeltaY(deltaY)
{
	// this constructor is for internal use only
}

///////////////////////////////////////////////////////////////////////////////

inline const BresenhamLineIterator& BresenhamLineIterator::operator++()
{
	mNum += mNumInc;
	if (mNum >= mDen)
	{
		mNum -= mDen;
		mX += mXInc1;
		mY += mYInc1;
	}
	mX += mXInc2;
	mY += mYInc2;

	++mI;

	return *this;
}

///////////////////////////////////////////////////////////////////////////////

inline const BresenhamLineIterator& BresenhamLineIterator::operator--()
{
	mNum -= mNumInc;
	if (mNum < 0)
	{
		mNum += mDen;
		mX -= mXInc1;
		mY -= mYInc1;
	}
	mX -= mXInc2;
	mY -= mYInc2;

	--mI;

	return *this;
}

///////////////////////////////////////////////////////////////////////////////

inline bool BresenhamLineIterator::hasNext() const
{
	if(mI<=mDen)
		return true;
	else
		return false;
}

///////////////////////////////////////////////////////////////////////////////

// normal line has tangent of dY/dX -> orthogonal line has tangent of -dX/dY.
// So we have to switch and negate the increments and switch the deltas. The
// line itself still starts at (mX,mY).
inline BresenhamLineIterator BresenhamLineIterator::orthogonal()
{
	return BresenhamLineIterator(mX, mY, mYInc1, mYInc2, -mXInc1, -mXInc2,
	                             mNum, mDen, mNumInc, mDeltaY, mDeltaX);
}

///////////////////////////////////////////////////////////////////////////////
// interface implementation

template<typename Visitor>
inline void bresenham(int x0, int y0, int x1, int y1, Visitor&& visitor)
{
	for(BresenhamLineIterator l(x0,y0,x1,y1); l.hasNext(); ++l)
		visitor(l.x(), l.y());

}

///////////////////////////////////////////////////////////////////////////////

/**
 * Detailed information on Bresenhams run-slice algorithm and source code
 * can be found here:
 *     http://www.phatcode.net/res/224/files/html/ch36/36-01.html
 *
 * @param[in] x0 : the x-coordinate of the first point
 * @param[in] y0 : the y-coordinate of the first point
 * @param[in] x1 : the x-coordinate of the second point
 * @param[in] y1 : the y-coordinate of the second point
 * @param[in] visitor : the visitor which has to provide the slice concept
 */

///////////////////////////////////////////////////////////////////////////////

template<typename Visitor>
inline void bresenhamRunSlice(int x0, int y0, int x1, int y1, Visitor&& visitor)
{
	// make sure we always go from top to bottom
	if (y1 < y0) {
		std::swap(y0,y1);
		std::swap(x0,x1);
	}

	int dx = x1 - x0;
	int dy = y1 - y0;

	int x, y;
	x = x0;
	y = y0;

	int adjUp, adjDown, error;
	int wholeStep, initialLength, finalLength, runLength;

	// are we going from right to left ?
	if (dx < 0)
		dx = -dx;

	// special case: horizontal line
	if(dy==0) {
		if(x0<=x1)
			visitor(x0, y0, dx+1);
		else
			visitor(x1, y0, dx+1);
		return;
	}

	// use normal bresenham for lines with major y-axis
	if(dx<=dy) {
		for(BresenhamLineIterator l(x0,y0,x1,y1); l.hasNext(); ++l)
			visitor(l.x(), l.y(), 1);
		return;
	}

	// use run-slice algorithm for lines with major x-axis:

	// minimum number of pixels in a run in this line
	wholeStep = dx / dy;

	// Error term adjust each time Y steps by 1; used to tell when one
	// extra pixel should be drawn as part of a run, to account for
	// fractional steps along the X axis per 1-pixel steps along Y
	adjUp = (dx % dy) * 2;

	// Error term adjust when the error term turns over, used to factor
	// out the X step made at that time
	adjDown = dy * 2;

	// Initial error term; reflects an initial step of 0.5 along the Y
	// axis
	error = (dx % dy) - (dy * 2);

	// The initial and last runs are partial, because Y advances only 0.5
	// for these runs, rather than 1. Divide one full run, plus the
	// initial pixel, between the initial and last runs
	initialLength = (wholeStep / 2) + 1;
	finalLength = initialLength;

	// If the basic run length is even and there's no fractional
	// advance, we have one pixel that could go to either the initial
	// or last partial run, which we'll arbitrarily allocate to the
	// last run
	if ((adjUp == 0) && ((wholeStep & 0x01) == 0)) {
		initialLength--;
	}

	// If there're an odd number of pixels per run, we have 1 pixel that can't
	// be allocated to either the initial or last partial run, so we'll add 0.5
	// to error term so this pixel will be handled by the normal full-run loop
	if ((wholeStep & 0x01) != 0) {
		error += dy;
	}

	if (x1 >= x0) {
		// Draw the first, partial run of pixels
		visitor(x, y, initialLength);
		x += initialLength;
		++y;
		///////////////////////////////////////////////////////////////////

		// process full runs
		for (int i = 0; i < (dy - 1); i++) {
			runLength = wholeStep; // run is at least this long
			// advance the error term and add an extra pixel if the error
			// term so indicates
			if ((error += adjUp) > 0) {
				runLength++;
				error -= adjDown;
			}

			// call the visitor
			visitor(x, y, runLength);
			x += runLength;
			++y;
		}

		// the final run of pixels
		visitor(x, y, finalLength);
	} else {
		++x;

		// Draw the first, partial run of pixels
		x -= initialLength;
		visitor(x, y, initialLength);
		++y;

		///////////////////////////////////////////////////////////////////

		// process full runs
		for (int i = 0; i < (dy - 1); i++) {
			runLength = wholeStep; // run is at least this long
			// advance the error term and add an extra pixel if the error
			// term so indicates
			if ((error += adjUp) > 0) {
				runLength++;
				error -= adjDown;
			}

			// call the visitor
			x -= runLength;
			visitor(x, y, runLength);
			++y;
		}

		// the final run of pixels
		x -= finalLength;
		visitor(x, y, finalLength);
	}
}

///////////////////////////////////////////////////////////////////////////////
// implementation section: GeneralBresenhamLineIterator class implementation
///////////////////////////////////////////////////////////////////////////////
template <int D, typename T, int Drive, int Res>
inline GeneralBresenhamLineIterator<D, T, Drive, Res>::
GeneralBresenhamLineIterator(Point<T, D> start,
                             Point<T, D> end)
{
	init(start, end);
}

template <int D, typename T, int Drive, int Res>
inline void GeneralBresenhamLineIterator<D, T, Drive, Res>::
init(Point<T, D> start, Point<T, D> end)
{
	int64_t maxd = -1;

	for (int n = 0; n < D; ++n)
	{
		Axis& axis = mAxes[n];
		axis.startExact = start(n);
		axis.endExact   = end(n);
		axis.distance   = std::abs(Res * (axis.endExact - axis.startExact));
		axis.current    = floor(axis.startExact);
		axis.end        = floor(axis.endExact);
		axis.step       = (axis.endExact >= axis.startExact) ? 1 : -1;

		this->checkForDrivingAxis(n, axis.distance, maxd);
	}

	Axis& drive = mAxes[this->drivingAxis()];
	drive.distance   = std::ceil(std::abs(Res * (drive.endExact - drive.startExact)));

	T driveOffset = drive.startExact - drive.current;
	if (drive.step < 0)
		driveOffset = 1 - driveOffset;

	for (int n = 0; n < D; ++n)
	{
		Axis& axis  = mAxes[n];

		if (n == this->drivingAxis())
		{
			axis.error = 0;
			continue;
		}

		T axisOffset = axis.startExact -  axis.current;
		if (axis.step < 0)
			axisOffset = 1 - axisOffset;

		axis.error = floor(  axis.distance * (1 - driveOffset)
		                    -drive.distance * (1 - axisOffset) );
	}
}

template <int D, typename T, int Drive, int Res>
inline const GeneralBresenhamLineIterator<D, T, Drive, Res>&
GeneralBresenhamLineIterator<D, T, Drive, Res>::operator++()
{
	for (int n = 0; n < D; ++n)
	{
		Axis& axis = mAxes[n];

		if (n == this->drivingAxis())
			axis.current += axis.step;
		else
			this->step(axis, mAxes[this->drivingAxis()]);
	}

	return *this;
}

template <int D, typename T, int Drive, int Res>
inline Point<int, D>
GeneralBresenhamLineIterator<D, T, Drive, Res>::pos() const
{
	Point<int, D> p;
	for (int n = 0; n < D; ++n)
		p(n) = mAxes[n].current;

	return p;
}

///////////////////////////////////////////////////////////////////////////////
}

#endif
