/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VacantClass.h
 *    $Interface for the hollow class, which loads the library associated with
 *    the class if necessary.$.
 *
 * @author Ronny Stricker
 * @date   2010/10/21
 */

#ifndef _MIRA_VACANTCLASS_H_
#define _MIRA_VACANTCLASS_H_

#include <factory/Class.h>
#include <utils/SharedLibrary.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

// *****************************************************************************
// *** VacantClass
// *****************************************************************************

/**
 * @brief The VacantClass object is the implementation of the Class class for
 * classes which are NOT available since the corresponding library is NOT loaded.
 * Therefore, every call of the defined functions involves the loading of the
 * corresponding library.
 */
class VacantClass : public Class {

	template <typename Derived>
	friend class Deserializer;
	friend class ClassFactory;

	friend class ManifestAgent;

public:

	/**
	 * @brief Return unique id for the class.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 */
	virtual int getTypeId() const;

	/**
	 * @brief Returns the platform independent C++ typename of the class.
	 */
	virtual Typename getTypename() const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 * @see Class::newInstance()
	 */
	virtual Object* newInstance() const;

	/**
	 * @brief Return a new instance of the class associated with the class object.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 * @see Class::newInstance( int paramCount, ... )
	 */
	virtual Object* newInstance( int paramCount, ... ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 * @see Class::newInstance( std::string const& childIdentifier )
	 */
	virtual Object* newInstance( std::string const& childIdentifier ) const;

	/**
	 * @brief Return a new instance of the child class with the given identifier.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 * @see Class::newInstance( std::string const& childIdentifier, int paramCount, ... )
	 */
	virtual Object* newInstance( std::string const& childIdentifier,
	                             int paramCount, ... ) const;

	/**
	 * @brief Return true if the associated class is abstract.
	 * @throw XLogical if the corresponding lib cannot be loaded
	 */
	virtual bool isAbstract() const
	{
		return mIsAbstract;
	}

	MIRA_SPLIT_REFLECT_MEMBER

	/**
	 * @brief Implementation of class member reflection.
	 * Attention: the abstract keyword will be evaluated
	 * (in contrast to the default Class::reflectWrite)
	 */
	template<typename Reflector>
	void reflectWrite(Reflector& r);

	template<typename Reflector>
	void reflectRead(Reflector& r)
	{
		Class::reflectRead( r );
	}

protected:
	/**
	 * @brief This function should never be called
	 * @throw XLogical always
	 */
	virtual Object* newVAInstance( int paramCount, std::va_list ) const;

private:
	VacantClass() : Class("","",false), mLoadInitiated(false) {}

	VacantClass( std::string const& identifier,
	             std::string const& name,
	             std::map<std::string, std::string> const& metaInfo,
	             bool isAbstract,
	             std::string const& libraryPath,
	             bool libLoaded = false ) :
		Class( identifier, name, metaInfo, libLoaded ),
		mLoadInitiated(false), mIsAbstract(isAbstract)
	{
		mLib = libraryPath;
	}

private:
	/**
	 *  indicates if some function has already initiated the loading 
	 *  of the associated lib
	 */
	mutable bool mLoadInitiated;
	/// indicates if the real class is of abstract type
	bool mIsAbstract;	

	static SharedLibraryLoader& getLibraryLoader();
};


template<typename Reflector>
inline void VacantClass::reflectWrite(Reflector& r)
{
	Class::reflectWrite( r );
	r.member( "Abstract", mIsAbstract, "" );
}

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_VACANTCLASS_H_ */

