/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SignalHandler.h
 *    Registration of signal handlers for several interrupt signals.
 *
 * @author Erik Einhorn
 * @date   2010/10/07
 */

#ifndef _MIRA_SIGNALHANDLER_H_
#define _MIRA_SIGNALHANDLER_H_

#include <signal.h> // valid for both unix and windows

#ifndef Q_MOC_RUN
#include <boost/function.hpp>
#endif

#include <error/CallStack.h>
#include <platform/Platform.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Structure used in signal handlers to pass the signal and callstack
 * @ingroup ExceptionModule
 */
struct IntSignal
{
	IntSignal(int s, const CallStack& stack) :
		sig(s), callStack(stack) {}

	/// The signal that was raised (SIGABRT etc.)
	int sig;
	/// The call stack backtrace at the moment the signal was raised.
	const CallStack& callStack;
};

/**
 * Registers a function for the specified interrupt signal.
 * The function can be a static member function or a member function that is
 * binded using boost::bind().
 * The specified function must take a single int parameter where the signal
 * is passed.
 *
 * Example:
 * @code
 * class MyClass
 * {
 * public:
 *     MyClass() {
 *         installSignalHandler(SIGINT, boost::bind(&MyClass::myHandler, this, _1));
 *     }
 *
 * private:
 *
 *     void myHandler(const IntSignal& signal)
 *     {
 *         // this is called if SIGINT signal occurs
 *         // signal.sig contains SIGINT
 *         // signal.callStack contains a backtrace of the callstack
 *     }
 * };
 * @endcode
 *
 * @param sig      The interupt signal
 * @param function The function that should be called if the signal occurs.
 *
 * To ensure compatibility with Windows the signal should be one of:
 * - SIGABRT: Abnormal termination
 * - SIGFPE:  Floating-point error
 * - SIGILL:  Illegal instruction
 * - SIGINT:  CTRL+C signal
 * - SIGSEGV: Illegal storage access
 * - SIGTERM: Termination request
 *
 * @ingroup ExceptionModule
 */
void installSignalHandler(int sig, boost::function<void (const IntSignal&)> function);

///////////////////////////////////////////////////////////////////////////////

}

#endif
