/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogTimer.h
 *    Stop watch timer for logging execution times.
 *
 * @author Tim Langner
 * @date   2010/07/07
 */

#ifndef _MIRA_LOGTIMER_H_
#define _MIRA_LOGTIMER_H_

#include <error/LoggingCore.h>
#include <error/LoggingCore.h>
#include <utils/PathFinderConstExpr.h>

namespace mira
{

///////////////////////////////////////////////////////////////////////////////

/**
 * Class that can be used as a stop watch to measure execution time of operations.
 * Never use this class directly use the R_LOGTIMER macro instead.
 * An instance of this class will automatically write a log entry on construction
 * and an entry on destruction counting the time between both.
 * e.g.
 * @code
 * {
 * 		MIRA_LOGTIMER(DEBUG, MyTimer, "I watch you")
 * 		callLengthyOperation();
 * }
 * @endcode
 * will produce
 * LOGENTRY MyTimer[I watch you] started.
 * LOGENTRY MyTimer[I watch you] ended taking 00:00:00.531772
 *
 * @ingroup LoggingModule
 */
class LogTimer
{
public:

	/// Constructor taking details for a log entry
	LogTimer(SeverityLevel level,
	         const char* name,
	         const Time& time,
	         const char* file, int line,
	         const char* function,
	         ThreadID threadID,
	         const std::string& caption);

	/// Destructor
	~LogTimer();

	/**
	 * Use this function to trace the time consumed since the start of this timer.
	 */
	Duration peek() const;

	/**
	 * This function can be called to end tracing at a defined point before destruction.
	 * If this function is called no log is created upon destruction.
	 */
	void end();

private:
	LogRecord mRecord;
	std::string mCaption;
	bool mEnded;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Macro for starting a LogTimer
 *
 * @ingroup LoggingModule
 */
#define MIRA_LOGTIMER(level, name, text)          \
		constexpr auto fileInMIRAPath##name = mira::chopMIRAPath(__FILE__); \
		LogTimer name(level, #name, Time::now(), \
		              fileInMIRAPath##name, __LINE__,        \
		              MIRA_FUNCTION, getCurrentThreadID(), text);

/**
 * Macro for peeking LogTimer class at a defined point
 *
 * @ingroup LoggingModule
 */
#define MIRA_PEEKLOGTIMER(name) name.peek();

/**
 * Macro for ending LogTimer class at a defined point
 *
 * @ingroup LoggingModule
 */
#define MIRA_ENDLOGTIMER(name) name.end();

///////////////////////////////////////////////////////////////////////////////

}

#endif
