/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgTutorial.C
 *    This example file contains the image tutorial source code.
 *    For detailed documentation to the example code see Img.dox file or the MIRA webpage.
 *	  Note: be aware that changing this file directly effects the tutorial written in Img.dox
 *
 *  @author Michael Volkhardt
 * 	@date   2010/11/20
 */

#include <geometry/Point.h>
#include <image/Img.h>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

int main() {
	//create an empty one channel 8-bit image with a size of 320x240 pixels:
	Img8U1 grayImage(320, 240); //Img8U1 is a typedef of Img<uint8,1>

	//access the image directly and set the pixel (10,20) to 255
	grayImage(10, 20) = 255; //sets pixel(10,20)

	// get direct access to a scanline of the image
	uint8* line = grayImage[40]; //access to scan line
	for (uint8 i = 10; i < 50; ++i) { //setting pixels in line to gray
		line[i] = 128;
	}
	cv::imshow("grayImage", (cv::Mat&)grayImage);

	// create an empty 3 channel image with 8 bit per channel and a size of 320x240 pixels:
	Img8U3 colorImg(320, 240); //Img8U3 is a typedef of Img<uint8,3>


	// set the pixel (10,20) to blue, note that OpenCV uses BGR image,
	// i.e. the blue channel is stored in the 3rd channel:
	colorImg(10, 20) = Img8U3::Pixel(255, 0, 0); // note that OpenCV uses BGR by default

	// use the Color-classes to specify the color to set a pixel with red color
	colorImg(10, 25) = Color::RGB(1.0f, 0.0f, 0.0f); //specify color space and set pixel (red)
	colorImg(10, 30) = Color::HSV(0.5f, 0.8f, 0.9f); //setting pixel in HSV color space

	//direct access to a scanline of the image
	Img8U3::Pixel* colorLine = colorImg[40];
	for (int i = 10; i < 50; ++i) { //drawing a yellow line at y=40
		colorLine[i] = Color::Yellow;
	}
	for (int i = 20; i < 30; ++i) { //red line on top of yellow one
		colorLine[i][1] = 0; //setting second channel of pixels to zero
	}

	// ImgXYZ types can be casted into cv::Mat matrices without
	// performance penalty or creating temporary copies
	cv::Mat cvmat = colorImg;

	// use OpenCV functions to manipulate the image:
	cv::line((cv::Mat&)colorImg, Point2i(10, 60), Point2i(50, 80), Color::Blue); //draw blue line
	cv::circle((cv::Mat&)colorImg, Point2i(30, 70), 10, Color::Lab(100, 100, -50)); // draw Lab circle
	cv::imshow("colorImg", (cv::Mat&)colorImg);

	cv::imwrite("img.png", (cv::Mat&)colorImg);
	cv::Mat tMat = cv::imread("img.png", -1); //unknown image type and channels
	Img<> tImg(tMat); //create untyped image
	cv::imshow("untyped Img", (cv::Mat&)tImg);

	cv::waitKey();

	return 0;
}
