/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PersistentError.C
 *    More detailed example of working with SQLite databases.
 *
 * @author Tim Langner
 * @date   2010/11/05
 */

#include <utils/Time.h>
#include <utils/MakeString.h>

#include <database/SQLiteDB.h>

using namespace mira;
using namespace std;

/**
 * Takes error messages that should be stored persistently in a database
 * and stores them in a SQLite database. The error messages can be translated into
 * different languages. It will also count how often an error occurs.
 */
class PersistentError
{
public:
	PersistentError();

	void setError(const std::string& error, const Duration& difference);
	void resetError(const std::string& error);
	std::string getErrorText(const std::string& error, const std::string& language);

private:
	SQLiteDB mDB;
};

PersistentError::PersistentError() :
	mDB("error.db")
{
	// create the error table that holds all persistent errors
	if ( !mDB.tableExists("error") )
		mDB.exec("create table error(id integer primary key, errno text, count integer, last integer);");

	// create translation tables for error texts
	if ( !mDB.tableExists("error_english") )
	{
		mDB.exec("create table error_english(errno text, error_text text);");
		mDB.exec("insert into error_english values ('SICK defect','The SICK laser scanner is defect');");
	}

	if ( !mDB.tableExists("error_deutsch") )
	{
		mDB.exec("create table error_deutsch(errno text, error_text text);");
		mDB.exec("insert into error_deutsch values ('SICK defect','Der SICK Laserscanner ist defekt');");
	}
}

void PersistentError::setError(const std::string& error, const Duration& difference)
{
	SQLiteQuery q = mDB.query("select * from error where errno = '" + 
	                          error + "' order by last desc;");
	Time now = Time::now();
	std::cout << "now = " << now << " and " << now.toUnixNS() << std::endl ;
	// the given error does not already  exist so insert it into the database 
	if ( q.eof() )
	{
		std::cout << "Inserting new because empty" << std::endl;
		mDB.exec(MakeString() << "insert into error (errno, count, last) values ('"
				<< error << "', 1, "  << now.toUnixNS() << ");");
	}
	// error does exist 
	else
	{
		// get last time the error was reported
		Time last = Time::fromUnixNS(q.getValue<int64>(3));
		// the time does not exceed the given duration 
		// so update its count and the time it was last encountered
		if ( (now-last) < difference )
			mDB.exec(MakeString() << "update error set count = "
					<< (q.getValue<int>(2) + 1)
					<< ", last = " << now.toUnixNS()
					<< " where errno = '" << error << "' and id = " <<
					q.getValue<int>(0) << ";");
		else
			// last time the error was reported was too long ago so add a new entry
			mDB.exec(MakeString() << "insert into error (errno, count, last) values ('"
					<< error + "', 1, " << now.toUnixNS() << ");");
	}
}

void PersistentError::resetError(const std::string& error)
{
	mDB.exec("delete from error where errno = '" + error + "';");
}

std::string PersistentError::getErrorText(const std::string& error, 
                                          const std::string& language)
{
	// try to translate an error text into the given language
	// by accessing the correct language table
	std::string table = "error_"+language;
	if ( !mDB.tableExists(table) )
		return error;
	SQLiteQuery q = mDB.query("select error_text from " + table + 
	                          " where errno = '" + error + "';");
	if ( q.eof() )
		return error;
	return q.getValue<std::string>(0);
}

int main(int argc, char** argv)
{
	PersistentError pe;
	// report an error about defect laser
	pe.setError("SICK defect", Duration::seconds(10));

	SQLiteDB db("error.db");
	// number of reported errors
	std::cout << "Table error contains rows " << 
		db.query("select count(*) from error;").getValue<int>(0) << std::endl;

	// Do we have laser defects
	std::cout << "We have Sick errors " << 
		db.query("select total(error.count) from error where errno='SICK defect';").getValue<float>(0) << std::endl;

	// plot the whole content of our error database
	SQLiteQuery q = db.query("select * from error order by 4;");
	for (int i=0; i<q.getColumnCount(); ++i)
	{
		std::cout << q.getName(i);
		if ( i < q.getColumnCount()-1 )
			std::cout << "\t|\t";
	}
	std::cout << std::endl;
	while ( !q.eof() )
	{
		std::cout << q.getValue<int>(0) << "\t|\t";
		if ( argc == 2 )
			std::cout << pe.getErrorText(q.getValue<std::string>(1), argv[1]) << "\t|\t";
		else
			std::cout << q.getValue<std::string>(1) << "\t|\t";
		std::cout << q.getValue<int>(2) << "\t|\t"
				<< Time::fromUnixNS(q.getValue<int64>(3)) << std::endl;
		++q;
	}
	return 0;
}
